/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/chattypes.h"

#include "twitchsdk/core/stringutilities.h"

ttv::chat::FeatureFlags::FeatureFlags() : conversations(false) {}

ttv::chat::FeatureFlags ttv::chat::FeatureFlags::All() {
  FeatureFlags flags;
  flags.conversations = true;

  return flags;
}

ttv::chat::FeatureFlags ttv::chat::FeatureFlags::None() {
  return FeatureFlags();
}

ttv::chat::TokenizationOptions::TokenizationOptions() : emoticons(false), mentions(false), urls(false), bits(false) {}

ttv::chat::TokenizationOptions ttv::chat::TokenizationOptions::All() {
  TokenizationOptions options;
  options.emoticons = true;
  options.mentions = true;
  options.urls = true;
  options.bits = true;

  return options;
}

ttv::chat::TokenizationOptions ttv::chat::TokenizationOptions::None() {
  return TokenizationOptions();
}

bool ttv::chat::TokenizationOptions::Any() const {
  return emoticons || mentions || urls || bits;
}

ttv::chat::ChatModeInfo::ChatModeInfo() : slowModeDurationSeconds(0), r9kMode(false), emotesOnlyMode(false) {}

ttv::chat::UserMode::UserMode()
    : moderator(false),
      broadcaster(false),
      administrator(false),
      staff(false),
      system(false),
      globalModerator(false),
      banned(false),
      subscriber(false),
      vip(false) {}

bool ttv::chat::UserMode::operator==(const UserMode& rhs) const {
  return ((moderator == rhs.moderator) && (broadcaster == rhs.broadcaster) && (administrator == rhs.administrator) &&
          (staff == rhs.staff) && (system == rhs.system) && (globalModerator == rhs.globalModerator) &&
          (banned == rhs.banned) && (subscriber == rhs.subscriber) && (vip == rhs.vip));
}

bool ttv::chat::UserMode::operator!=(const UserMode& rhs) const {
  return !(*this == rhs);
}

ttv::chat::ChatUserInfo::ChatUserInfo() : nameColorARGB(0xFF000000), userId(0) {}

bool ttv::chat::ChatUserInfo::operator==(const ChatUserInfo& rhs) const {
  return ((userName == rhs.userName) && (displayName == rhs.displayName) && (nameColorARGB == rhs.nameColorARGB) &&
          (userId == rhs.userId) && (userMode == rhs.userMode));
}

bool ttv::chat::ChatUserInfo::operator!=(const ttv::chat::ChatUserInfo& rhs) const {
  return !(*this == rhs);
}

ttv::chat::RestrictionReason::RestrictionReason()
    : anonymous(false), subscribersOnly(false), slowMode(false), timeout(false), banned(false) {}

bool ttv::chat::RestrictionReason::operator==(const RestrictionReason& rhs) const {
  return ((anonymous == rhs.anonymous) && (subscribersOnly == rhs.subscribersOnly) && (slowMode == rhs.slowMode) &&
          (timeout == rhs.timeout) && (banned == rhs.banned));
}

bool ttv::chat::RestrictionReason::operator!=(const RestrictionReason& rhs) const {
  return !(*this == rhs);
}

ttv::chat::ChatChannelRestrictions::ChatChannelRestrictions()
    : followersDuration(0),
      slowModeDuration(0),
      slowModeSetAt(0),
      emoteOnly(false),
      verifiedOnly(false),
      followersOnly(false),
      subscribersOnly(false),
      slowMode(false),
      r9k(false) {}

ttv::chat::AutoModFlags::AutoModFlags() : identityLevel(0), sexualLevel(0), aggressiveLevel(0), profanityLevel(0) {}

bool ttv::chat::AutoModFlags::HasFlag() {
  return (identityLevel != 0 || sexualLevel != 0 || aggressiveLevel != 0 || profanityLevel != 0);
}

bool ttv::chat::TokensEqual(const std::unique_ptr<MessageToken>& lhs, const std::unique_ptr<MessageToken>& rhs) {
  if (lhs->GetType() != rhs->GetType()) {
    return false;
  }

  switch (lhs->GetType()) {
    case MessageToken::Type::Text: {
      auto textLhs = static_cast<TextToken*>(lhs.get());
      auto textRhs = static_cast<TextToken*>(rhs.get());
      return textLhs->text == textRhs->text;
    }

    case MessageToken::Type::Emoticon: {
      auto emoticonLhs = static_cast<EmoticonToken*>(lhs.get());
      auto emoticonRhs = static_cast<EmoticonToken*>(rhs.get());
      return (emoticonLhs->emoticonId == emoticonRhs->emoticonId) &&
             (emoticonLhs->emoticonText == emoticonRhs->emoticonText);
    }

    case MessageToken::Type::Mention: {
      auto mentionLhs = static_cast<MentionToken*>(lhs.get());
      auto mentionRhs = static_cast<MentionToken*>(rhs.get());

      // Don't compare color because it can be selected randomly.
      return (mentionLhs->userName == mentionRhs->userName);
    }

    case MessageToken::Type::Url: {
      auto urlLhs = static_cast<UrlToken*>(lhs.get());
      auto urlRhs = static_cast<UrlToken*>(rhs.get());
      return (urlLhs->hidden == urlRhs->hidden) && (urlLhs->url == urlRhs->url);
    }

    case MessageToken::Type::Bits: {
      auto bitsLhs = static_cast<BitsToken*>(lhs.get());
      auto bitsRhs = static_cast<BitsToken*>(rhs.get());
      return (bitsLhs->numBits == bitsRhs->numBits) && (bitsLhs->prefix == bitsRhs->prefix);
    }
  }

  return false;
}

ttv::chat::TextToken::TextToken(const std::string& textArg) : text(textArg) {}

ttv::chat::TextToken::TextToken(const std::string& textArg, const AutoModFlags& flags)
    : text(textArg), autoModFlags(flags) {}

ttv::chat::TextToken::TextToken(const TextToken& src) : text(src.text) {}

ttv::chat::TextToken& ttv::chat::TextToken::operator=(const TextToken& src) {
  text = src.text;
  autoModFlags = src.autoModFlags;
  return *this;
}

ttv::chat::TextToken::TextToken(TextToken&& src)
    : text(std::move(src.text)), autoModFlags(std::move(src.autoModFlags)) {}

ttv::chat::TextToken& ttv::chat::TextToken::operator=(TextToken&& src) {
  text = std::move(src.text);
  autoModFlags = std::move(src.autoModFlags);
  return *this;
}

ttv::chat::EmoticonToken::EmoticonToken(const std::string& emoticonTextArg, const std::string& emoticonIdArg)
    : emoticonText(emoticonTextArg), emoticonId(emoticonIdArg) {}

ttv::chat::EmoticonToken::EmoticonToken(const EmoticonToken& src)
    : emoticonText(src.emoticonText), emoticonId(src.emoticonId) {}

ttv::chat::EmoticonToken& ttv::chat::EmoticonToken::operator=(const EmoticonToken& src) {
  emoticonText = src.emoticonText;
  emoticonId = src.emoticonId;
  return *this;
}

ttv::chat::EmoticonToken::EmoticonToken(EmoticonToken&& src)
    : emoticonText(std::move(src.emoticonText)), emoticonId(src.emoticonId) {}

ttv::chat::EmoticonToken& ttv::chat::EmoticonToken::operator=(EmoticonToken&& src) {
  emoticonText = std::move(src.emoticonText);
  emoticonId = std::move(src.emoticonId);
  return *this;
}

ttv::chat::MentionToken::MentionToken(const std::string& userNameArg, const std::string& textArg, bool isLocalUserArg)
    : userName(userNameArg), text(textArg), isLocalUser(isLocalUserArg) {}

ttv::chat::MentionToken::MentionToken(const MentionToken& src)
    : userName(src.userName), text(src.text), isLocalUser(src.isLocalUser) {}

ttv::chat::MentionToken& ttv::chat::MentionToken::operator=(const MentionToken& src) {
  userName = src.userName;
  text = src.text;
  isLocalUser = src.isLocalUser;
  return *this;
}

ttv::chat::MentionToken::MentionToken(MentionToken&& src)
    : userName(std::move(src.userName)), text(std::move(src.text)), isLocalUser(src.isLocalUser) {}

ttv::chat::MentionToken& ttv::chat::MentionToken::operator=(MentionToken&& src) {
  userName = std::move(src.userName);
  text = std::move(src.text);
  isLocalUser = src.isLocalUser;
  return *this;
}

ttv::chat::UrlToken::UrlToken() : hidden(false) {}

ttv::chat::UrlToken::UrlToken(const std::string& urlArg, bool hiddenArg) : url(urlArg), hidden(hiddenArg) {}

ttv::chat::UrlToken::UrlToken(const UrlToken& src) : url(src.url), hidden(src.hidden) {}

ttv::chat::UrlToken& ttv::chat::UrlToken::operator=(const UrlToken& src) {
  url = src.url;
  hidden = src.hidden;
  return *this;
}

ttv::chat::UrlToken::UrlToken(UrlToken&& src) : url(std::move(src.url)), hidden(src.hidden) {}

ttv::chat::UrlToken& ttv::chat::UrlToken::operator=(UrlToken&& src) {
  url = std::move(src.url);
  hidden = src.hidden;
  return *this;
}

ttv::chat::BitsToken::BitsToken() : numBits(0) {}

ttv::chat::BitsToken::BitsToken(const std::string& prefixArg, uint32_t numBitsArg)
    : prefix(prefixArg), numBits(numBitsArg) {}

ttv::chat::BitsToken::BitsToken(const BitsToken& src) : prefix(src.prefix), numBits(src.numBits) {}

ttv::chat::BitsToken& ttv::chat::BitsToken::operator=(const BitsToken& src) {
  prefix = src.prefix;
  numBits = src.numBits;
  return *this;
}

ttv::chat::BitsToken::BitsToken(BitsToken&& src) : prefix(std::move(src.prefix)), numBits(src.numBits) {}

ttv::chat::BitsToken& ttv::chat::BitsToken::operator=(BitsToken&& src) {
  prefix = std::move(src.prefix);
  numBits = src.numBits;
  return *this;
}

ttv::chat::MessageInfo::Flags::Flags()
    : action(false), notice(false), ignored(false), deleted(false), containsBits(false) {}

bool ttv::chat::MessageInfo::Flags::operator==(const Flags& rhs) const {
  return ((action == rhs.action) && (notice == rhs.notice) && (ignored == rhs.ignored) && (deleted == rhs.deleted) &&
          (containsBits == rhs.containsBits));
}

bool ttv::chat::MessageInfo::Flags::operator!=(const ttv::chat::MessageInfo::Flags& rhs) const {
  return !(*this == rhs);
}

ttv::chat::MessageInfo::MessageInfo() : nameColorARGB(0xFF000000), timestamp(0), userId(0), numBitsSent(0) {}

ttv::chat::MessageInfo::MessageInfo(const MessageInfo& src) {
  *this = src;
}

ttv::chat::MessageInfo& ttv::chat::MessageInfo::operator=(const ttv::chat::MessageInfo& src) {
  userName = src.userName;
  displayName = src.displayName;
  badges = src.badges;
  userMode = src.userMode;
  flags = src.flags;
  nameColorARGB = src.nameColorARGB;
  timestamp = src.timestamp;
  userId = src.userId;
  numBitsSent = src.numBitsSent;
  messageTags = src.messageTags;
  messageType = src.messageType;

  tokens.clear();
  std::transform(src.tokens.begin(), src.tokens.end(), std::back_inserter(tokens),
    [](const std::unique_ptr<MessageToken>& token) { return token->Clone(); });

  return *this;
}

ttv::chat::WhisperMessage::WhisperMessage() : messageId(0) {}

ttv::chat::ExtensionMessage::ExtensionMessage() : chatColor(0xFF000000), sentAt(0) {}

ttv::chat::SubscriptionNotice::SubscriptionNotice()
    : subStreakMonthCount(0),
      subCumulativeMonthCount(0),
      senderCount(0),
      massGiftCount(0),
      benefitEndMonth(0),
      type(Type::Unknown),
      plan(Plan::Unknown),
      shouldShowSubStreak(false) {}

ttv::chat::SubscriptionNotice::SubscriptionNotice(const SubscriptionNotice& source) {
  *this = source;
}

ttv::chat::SubscriptionNotice& ttv::chat::SubscriptionNotice::operator=(const SubscriptionNotice& source) {
  userMessage = (source.userMessage != nullptr) ? std::make_unique<MessageInfo>(*source.userMessage) : nullptr;
  systemMessage = source.systemMessage;
  planDisplayName = source.planDisplayName;
  messageId = source.messageId;
  recipient = source.recipient;
  subStreakMonthCount = source.subStreakMonthCount;
  subCumulativeMonthCount = source.subCumulativeMonthCount;
  senderCount = source.senderCount;
  massGiftCount = source.massGiftCount;
  benefitEndMonth = source.benefitEndMonth;
  type = source.type;
  plan = source.plan;
  shouldShowSubStreak = source.shouldShowSubStreak;

  return *this;
}

ttv::chat::SubscriptionNotice::Recipient::Recipient() : userId(0) {}

ttv::chat::RaidNotice::RaidNotice() : viewerCount(0) {}

ttv::chat::BadgeEntitlement::BadgeEntitlement() : newLevel(0), previousLevel(0), isNewBadgeLevel(false) {}

ttv::chat::BitsSentEvent::BitsSentEvent() : channelId(0), userBitsBalance(0), channelBitsTotal(0) {}

ttv::chat::ChatComment::ChatComment() : channelId(0), timestampMilliseconds(0), updatedAt(0), moreReplies(false) {}

ttv::chat::BadgeImage::BadgeImage() : scale(0.0) {}

ttv::chat::BadgeVersion::BadgeVersion() : clickAction(Action::None) {}

ttv::chat::Emoticon::Emoticon() : isRegex(false) {}

bool ttv::chat::Emoticon::operator==(const ttv::chat::Emoticon& rhs) const {
  return (match == rhs.match) && (emoticonId == rhs.emoticonId) && (isRegex == rhs.isRegex);
}

bool ttv::chat::EmoticonSet::operator==(const ttv::chat::EmoticonSet& rhs) const {
  // Do not compare the emoticon set IDs because certain sets can have changing IDs.
  // Skipping the ID comparison is fine because emoticon IDs are still unique across sets.
  return (emoticons == rhs.emoticons);
}

ttv::chat::BitsConfiguration::CheermoteImage::CheermoteImage() : dpiScale(0), isAnimated(false) {}

ttv::chat::BitsConfiguration::CheermoteTier::CheermoteTier()
    : bits(0), color(0), canCheer(false), canShowInBitsCard(false) {}

ttv::chat::BitsConfiguration::Cheermote::Cheermote() : type(Type::Unknown) {}

ttv::chat::BitsConfiguration::BitsConfiguration() : mChannelId(0) {}

ttv::chat::BitsConfiguration::BitsConfiguration(
  const std::vector<Cheermote>& cheermotes, UserId userId, ChannelId channelId)
    : mCheermotes(cheermotes), mUserId(userId), mChannelId(channelId) {
  SortCheermoteTiers();
}

ttv::chat::BitsConfiguration::BitsConfiguration(std::vector<Cheermote>&& cheermotes, UserId userId, ChannelId channelId)
    : mCheermotes(std::move(cheermotes)), mUserId(userId), mChannelId(channelId) {
  SortCheermoteTiers();
}

TTV_ErrorCode ttv::chat::BitsConfiguration::GetBitsImageUrl(const std::string& prefix, uint32_t numBits,
  BitsConfiguration::CheermoteImage::Theme theme, float dpiScale, bool isAnimated, std::string& url,
  Color& color) const {
  url.clear();

  auto cheermoteIter = std::find_if(mCheermotes.begin(), mCheermotes.end(),
    [&prefix](const Cheermote& cheermote) { return strcasecmp(cheermote.prefix.c_str(), prefix.c_str()) == 0; });

  if (cheermoteIter != mCheermotes.end()) {
    for (auto tierIter = cheermoteIter->tiers.rbegin(); tierIter != cheermoteIter->tiers.rend(); ++tierIter) {
      if (numBits >= tierIter->bits) {
        color = tierIter->color;

        auto imageIter = std::find_if(
          tierIter->images.begin(), tierIter->images.end(), [isAnimated, theme, dpiScale](const CheermoteImage& image) {
            return (image.theme == theme) && (image.isAnimated == isAnimated) &&
                   (std::abs(image.dpiScale - dpiScale) < 0.001f);
          });

        if (imageIter != tierIter->images.end()) {
          url = imageIter->url;
          return TTV_EC_SUCCESS;
        }
      }
    }
  }

  return TTV_EC_NOT_AVAILABLE;
}

TTV_ErrorCode ttv::chat::BitsConfiguration::GetHighestDpiBitsImageUrl(const std::string& prefix, uint32_t numBits,
  BitsConfiguration::CheermoteImage::Theme theme, float dpiScaleLimit, bool isAnimated, std::string& url,
  Color& color) const {
  url.clear();

  auto cheermoteIter = std::find_if(mCheermotes.begin(), mCheermotes.end(),
    [&prefix](const Cheermote& cheermote) { return strcasecmp(cheermote.prefix.c_str(), prefix.c_str()) == 0; });

  if (cheermoteIter != mCheermotes.end()) {
    for (auto tierIter = cheermoteIter->tiers.rbegin(); tierIter != cheermoteIter->tiers.rend(); ++tierIter) {
      if (numBits >= tierIter->bits) {
        float maxDpi = 0;
        color = tierIter->color;

        for (const auto& image : tierIter->images) {
          if (image.theme == theme && image.isAnimated == isAnimated && image.dpiScale > maxDpi &&
              (image.dpiScale <= dpiScaleLimit || std::abs(image.dpiScale - dpiScaleLimit) < 0.001f)) {
            maxDpi = image.dpiScale;
            url = image.url;
          }
        }

        if (!url.empty()) {
          return TTV_EC_SUCCESS;
        }
      }
    }
  }

  return TTV_EC_NOT_AVAILABLE;
}

const std::vector<ttv::chat::BitsConfiguration::Cheermote>& ttv::chat::BitsConfiguration::GetCheermotes() const {
  return mCheermotes;
}

ttv::ChannelId ttv::chat::BitsConfiguration::GetUserId() const {
  return mUserId;
}

ttv::ChannelId ttv::chat::BitsConfiguration::GetChannelId() const {
  return mChannelId;
}

void ttv::chat::BitsConfiguration::SortCheermoteTiers() {
  for (auto& cheermote : mCheermotes) {
    // Sort the tiers by increasing minBits for sanity
    std::sort(cheermote.tiers.begin(), cheermote.tiers.end(),
      [](const BitsConfiguration::CheermoteTier& a, const BitsConfiguration::CheermoteTier& b) {
        return a.bits < b.bits;
      });
  }
}

ttv::chat::UnreadThreadCounts::UnreadThreadCounts() : unreadThreadCount(0), unreadMessageCount(0), exhaustive(false) {}

bool ttv::chat::UnreadThreadCounts::operator==(const ttv::chat::UnreadThreadCounts& rhs) const {
  return (unreadThreadCount == rhs.unreadThreadCount) && (unreadMessageCount == rhs.unreadMessageCount) &&
         (exhaustive == rhs.exhaustive);
}

bool ttv::chat::UnreadThreadCounts::operator!=(const ttv::chat::UnreadThreadCounts& rhs) const {
  return !(*this == rhs);
}

ttv::chat::UserList::UserList() : totalUserCount(0) {}

TTV_ErrorCode ttv::chat::BadgeVersion::FindImage(float scale, BadgeImage& result) const {
  if (images.empty()) {
    return TTV_EC_NOT_AVAILABLE;
  }

  if (scale <= (images.front().scale + std::numeric_limits<float>::epsilon())) {
    result = images.front();
  } else if (scale >= (images.back().scale - std::numeric_limits<float>::epsilon())) {
    result = images.back();
  }

  for (const auto& image : images) {
    if (scale <= (image.scale + std::numeric_limits<float>::epsilon())) {
      result = image;
      return TTV_EC_SUCCESS;
    }
  }

  return TTV_EC_NOT_AVAILABLE;
}

TTV_ErrorCode ttv::chat::BadgeSet::FindBadge(const MessageBadge& badge, BadgeVersion& result) const {
  auto badgeSetIter = badges.find(badge.name);
  if (badgeSetIter != badges.end()) {
    const auto& set = badgeSetIter->second;
    auto versionIter = set.versions.find(badge.version);
    if (versionIter != set.versions.end()) {
      result = versionIter->second;
      return TTV_EC_SUCCESS;
    }
  }

  return TTV_EC_NOT_AVAILABLE;
}

TTV_ErrorCode ttv::chat::BadgeSet::FindBadgeImage(const MessageBadge& badge, float scale, BadgeImage& result) const {
  BadgeVersion version;
  TTV_ErrorCode ec = FindBadge(badge, version);
  if (TTV_SUCCEEDED(ec)) {
    ec = version.FindImage(scale, result);
  }
  return ec;
}

ttv::chat::RoomRolePermissions::RoomRolePermissions() : read(RoomRole::Unknown), send(RoomRole::Unknown) {}

ttv::chat::RoomRolePermissions::RoomRolePermissions(RoomRole readArg, RoomRole sendArg)
    : read(readArg), send(sendArg) {}

ttv::chat::ChatRoomPermissions::ChatRoomPermissions() : readMessages(false), sendMessages(false), moderate(false) {}

ttv::chat::ChatRoomView::ChatRoomView()
    : lastReadAt(0), unreadMentionCount(0), isMuted(false), isArchived(false), isUnread(false) {}

ttv::chat::RoomMentionInfo::RoomMentionInfo() : roomOwnerId(0), senderId(0), sentAt(0) {}

ttv::chat::RaidStatus::RaidStatus()
    : creatorUserId(0),
      sourceChannelId(0),
      targetChannelId(0),
      numUsersInRaid(0),
      transitionJitterSeconds(0),
      forceRaidNowSeconds(0),
      joined(true) {}

bool ttv::chat::RaidStatus::operator==(const RaidStatus& other) const {
  return creatorUserId == other.creatorUserId && sourceChannelId == other.sourceChannelId &&
         targetChannelId == other.targetChannelId && targetUserLogin == other.targetUserLogin &&
         targetUserDisplayName == other.targetUserDisplayName &&
         targetUserProfileImageUrl == other.targetUserProfileImageUrl &&
         transitionJitterSeconds == other.transitionJitterSeconds && numUsersInRaid == other.numUsersInRaid &&
         forceRaidNowSeconds == other.forceRaidNowSeconds && joined == other.joined;
}

bool ttv::chat::RaidStatus::operator!=(const RaidStatus& other) const {
  return !(*this == other);
}

ttv::chat::ChannelVodCommentSettings::ChannelVodCommentSettings()
    : channelId(0), createdAt(0), updatedAt(0), followersOnlyDurationSeconds(0) {}

ttv::chat::MultiviewContentAttribute::MultiviewContentAttribute() : ownerChannelId(0), createdAt(0), updatedAt(0) {}

ttv::chat::Chanlet::Chanlet() : chanletId(0) {}
