/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatuserblocklist.h"

#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/task/chatchangeuserblocktask.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

namespace {
const char* kLoggerName = "ChatUserBlockList";
const uint64_t kDefaultRefreshInterval = 24 * 60 * 60 * 1000;  // Refresh every 24 hours
const uint64_t kFailureRefreshInterval = 60 * 1000;            // Refresh after ten minutes
}  // namespace

ttv::chat::ChatUserBlockList::ChatUserBlockList(std::shared_ptr<User> user)
    : UserComponent(user), mRefreshIntervalMilliseconds(kDefaultRefreshInterval) {
  Log(MessageLevel::Debug, "ChatUserBlockList()");

  mBlockedUsers = std::make_shared<std::unordered_set<UserId>>();
  mRefreshResult = std::make_shared<ChatGetBlockListTask::Result>();

  // Refresh immediately
  mRefreshTimer.Set(0);
}

ttv::chat::ChatUserBlockList::~ChatUserBlockList() {
  TTV_ASSERT(mState == State::Uninitialized);

  Log(MessageLevel::Debug, "~ChatUserBlockList()");
}

void ttv::chat::ChatUserBlockList::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  if (mState == State::Initialized) {
    if (!IsRefreshInProgress() && !mOAuthIssue) {
      // Time to update the list
      if (mRefreshTimer.Check(false)) {
        // Wait until the running change task is done
        if (mRunningChangeTask == nullptr) {
          mRefreshTimer.Clear();
          UpdateList();
        }
      } else {
        if (mRunningChangeTask == nullptr) {
          ProcessNextRequest();
        }
      }
    }
  }

  UserComponent::Update();
}

TTV_ErrorCode ttv::chat::ChatUserBlockList::Shutdown() {
  TTV_ErrorCode ec = UserComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    // All pending changes will be lost
    mQueuedChanges.clear();

    // Flushed blocked users callbacks
    mBlockedUsersCallbacks.Flush(TTV_EC_SHUTTING_DOWN, std::vector<UserInfo>());
  }

  return ec;
}

bool ttv::chat::ChatUserBlockList::CheckShutdown() {
  if (!Component::CheckShutdown()) {
    return false;
  }

  if (IsRefreshInProgress()) {
    return false;
  }

  return true;
}

std::string ttv::chat::ChatUserBlockList::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::ChatUserBlockList::UpdateList() {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  mRefreshTimer.Clear();

  if (IsRefreshInProgress()) {
    return TTV_EC_REQUEST_PENDING;
  }

  mRefreshUsers = std::make_shared<std::unordered_set<UserId>>();
  mRefreshResult = std::make_shared<ChatGetBlockListTask::Result>();
  mUserInfoList.clear();

  FetchBlocks();

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatUserBlockList::FetchBlocks() {
  auto currentUser = mUser.lock();
  if (currentUser == nullptr) {
    RefreshComplete(TTV_EC_NEED_TO_LOGIN);
    return;
  }

  Log(MessageLevel::Debug, "FetchPage");

  TTV_ASSERT(mTaskRunner != nullptr);

  mRefreshResult->valid = false;

  auto oauthToken = currentUser->GetOAuthToken();

  ChatGetBlockListTask::Callback callback = [this, currentUser, oauthToken](ChatGetBlockListTask* source,
                                              TTV_ErrorCode ec,
                                              std::shared_ptr<ChatGetBlockListTask::Result> /*result*/) {
    Log(MessageLevel::Debug, "OnBlockListRequestComplete ec: %s", ErrorToString(ec));

    CompleteTask(source);
    mRunningGetListTask.reset();

    if (TTV_SUCCEEDED(ec)) {
      if (mState == State::Initialized) {
        // Add the page of users to the final result
        for (const auto& user : mRefreshResult->users) {
          mRefreshUsers->insert(user.userId);
          mUserInfoList.emplace_back(user);
        }
      }
    } else if (ec == TTV_EC_AUTHENTICATION) {
      currentUser->ReportOAuthTokenInvalid(oauthToken, ec);
    }

    RefreshComplete(ec);
  };

  mRunningGetListTask =
    std::make_shared<ChatGetBlockListTask>(currentUser->GetUserId(), oauthToken->GetToken(), mRefreshResult, callback);

  TTV_ErrorCode ec = StartTask(mRunningGetListTask);

  // If the request wasn't submmitted then try again later
  if (TTV_FAILED(ec)) {
    RefreshComplete(TTV_EC_REQUEST_ABORTED);
    mRunningGetListTask.reset();
  }
}

void ttv::chat::ChatUserBlockList::RefreshComplete(TTV_ErrorCode ec) {
  if (TTV_SUCCEEDED(ec)) {
    mBlockedUsers = mRefreshUsers;
    mRefreshTimer.Set(mRefreshIntervalMilliseconds);
  } else {
    // Schedule a refresh sooner than we would under normal circumstances
    mRefreshTimer.Set(kFailureRefreshInterval);
  }

  mBlockedUsersCallbacks.Flush(ec, mUserInfoList);
  mRefreshUsers.reset();
  mRefreshResult.reset();
}

TTV_ErrorCode ttv::chat::ChatUserBlockList::BlockUser(
  UserId userId, const std::string& reason, bool whisper, const BlockChangeCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  CancelRequestsForUser(userId);

  ScheduleRequest(userId, true, reason, whisper, callback);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatUserBlockList::BlockUser(
  const std::string& username, const std::string& reason, bool whisper, const BlockChangeCallback& callback) {
  auto userRepository = mUserRepository.lock();
  if ((userRepository == nullptr) || (mState != State::Initialized)) {
    return TTV_EC_SHUT_DOWN;
  }

  UserInfo cachedUserInfo;
  if (TTV_SUCCEEDED(userRepository->GetUserInfoByName(username, cachedUserInfo))) {
    return BlockUser(cachedUserInfo.userId, reason, whisper, callback);
  }

  userRepository->FetchUserInfoByName(
    username, [this, reason, whisper, callback](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
      ErrorDetails details = errorDetails;

      if (TTV_SUCCEEDED(details.ec)) {
        details.ec = BlockUser(userInfo.userId, reason, whisper, callback);
      }

      if (TTV_FAILED(details.ec) && callback != nullptr) {
        callback(details.ec);
      }
    });

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatUserBlockList::UnblockUser(UserId userId, const BlockChangeCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  CancelRequestsForUser(userId);

  ScheduleRequest(userId, false, "", false, callback);

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatUserBlockList::UnblockUser(
  const std::string& username, const BlockChangeCallback& callback) {
  auto userRepository = mUserRepository.lock();
  if ((userRepository == nullptr) || (mState != State::Initialized)) {
    return TTV_EC_SHUT_DOWN;
  }

  UserInfo cachedUserInfo;
  if (TTV_SUCCEEDED(userRepository->GetUserInfoByName(username, cachedUserInfo))) {
    return UnblockUser(cachedUserInfo.userId, callback);
  }

  userRepository->FetchUserInfoByName(
    username, [this, callback](const ErrorDetails& errorDetails, const UserInfo& userInfo) {
      ErrorDetails details = errorDetails;

      if (TTV_SUCCEEDED(details.ec)) {
        details.ec = UnblockUser(userInfo.userId, callback);
      }

      if (TTV_FAILED(details.ec) && callback != nullptr) {
        callback(details.ec);
      }
    });

  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::ChatUserBlockList::FetchBlockedUsers(const FetchBlockedUsersCallback& callback) {
  TTV_RETURN_ON_DIFFERENT(mState, State::Initialized, TTV_EC_NOT_INITIALIZED);

  mBlockedUsersCallbacks.Push(callback);

  // Refresh immediately if we're not already refreshing
  if (!IsRefreshInProgress()) {
    mRefreshTimer.Set(0);
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::ChatUserBlockList::SetRefreshInterval(uint64_t milliseconds) {
  mRefreshIntervalMilliseconds = milliseconds;

  if (mRefreshTimer.IsSet()) {
    mRefreshTimer.AdjustDuration(milliseconds);
  }
}

bool ttv::chat::ChatUserBlockList::IsUserBlocked(UserId userId) {
  return mBlockedUsers->find(userId) != mBlockedUsers->end();
}

void ttv::chat::ChatUserBlockList::GetBlockedUsers(std::unordered_set<UserId>& result) const {
  result = *mBlockedUsers;
}

bool ttv::chat::ChatUserBlockList::IsRefreshInProgress() const {
  return mRefreshUsers != nullptr;
}

void ttv::chat::ChatUserBlockList::CancelRequestsForUser(UserId blockUserId) {
  for (auto iter = mQueuedChanges.begin(); iter != mQueuedChanges.end();) {
    const auto& request = *iter;

    if (request.userId == blockUserId) {
      iter = mQueuedChanges.erase(iter);
    } else {
      ++iter;
    }
  }
}

void ttv::chat::ChatUserBlockList::ScheduleRequest(
  UserId blockUserId, bool block, const std::string& reason, bool whisper, const BlockChangeCallback& callback) {
  ChangeRequest request;
  request.block = block;
  request.reason = reason;
  request.userId = blockUserId;
  request.whisper = whisper;
  request.callback = callback;

  mQueuedChanges.push_back(request);
}

void ttv::chat::ChatUserBlockList::ProcessNextRequest() {
  if (mQueuedChanges.empty()) {
    return;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    Log(MessageLevel::Debug, "ChatUserBlockList::ProcessNextRequest: No user");
    return;
  }

  auto oauthToken = user->GetOAuthToken();

  const auto& callback = mQueuedChanges[0].callback;
  ChatChangeUserBlockTask::Callback taskCallback = [this, user, oauthToken, callback](ChatChangeUserBlockTask* source,
                                                     TTV_ErrorCode ec, UserId userId, UserId blockUserId, bool block) {
    TTV_ASSERT(userId == user->GetUserId());

    CompleteTask(source);
    mRunningChangeTask.reset();

    Log(MessageLevel::Debug, "OnChangeBlockRequestComplete ec: %s", ErrorToString(ec));

    // Unignoring a user that wasn't ignored shouldn't be an error since ignoring a user that was already ignored isn't
    if (ec == TTV_EC_API_REQUEST_FAILED && !block && !IsUserBlocked(blockUserId)) {
      ec = TTV_EC_SUCCESS;
    }

    if (TTV_SUCCEEDED(ec)) {
      Log(MessageLevel::Debug, "OnChangeBlockRequestComplete: %lu %s succeeded", blockUserId,
        block ? "block" : "unblock");

      SetLocalValue(mBlockedUsers, blockUserId, block);
    } else {
      if (ec == TTV_EC_AUTHENTICATION) {
        user->ReportOAuthTokenInvalid(oauthToken, ec);
      }

      Log(MessageLevel::Debug, "OnChangeBlockRequestComplete: %lu %s failed: %s", blockUserId,
        block ? "block" : "unblock", ErrorToString(ec));
    }

    if (callback != nullptr) {
      callback(ec);
    }
  };

  mRunningChangeTask = std::make_shared<ChatChangeUserBlockTask>(user->GetUserId(), oauthToken->GetToken(),
    mQueuedChanges[0].userId, mQueuedChanges[0].block, mQueuedChanges[0].reason, taskCallback);

  TTV_ErrorCode ec = StartTask(mRunningChangeTask);
  if (TTV_SUCCEEDED(ec)) {
    mRunningChangeTask->SetWhisper(mQueuedChanges[0].whisper);
    mQueuedChanges.erase(mQueuedChanges.begin());
  } else {
    if (callback != nullptr) {
      callback(ec);
    }
    mRunningChangeTask.reset();
  }
}

void ttv::chat::ChatUserBlockList::SetLocalValue(SetReference set, UserId blockUserId, bool block) {
  auto iter = std::find(set->begin(), set->end(), blockUserId);

  if (block) {
    if (iter == set->end()) {
      set->insert(blockUserId);
    }
  } else {
    if (iter != set->end()) {
      set->erase(iter);
    }
  }
}
