/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatuserthreads.h"

#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/chat/internal/task/chatgetemoticonstask.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/chat/internal/useremoticonsets.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/types/errortypes.h"
#include "twitchsdk/core/user/oauthtoken.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

#include <algorithm>
#include <sstream>

namespace {
const char* kLoggerName = "ChatUserThreads";
}  // namespace

ttv::chat::ChatUserThreads::ChatUserThreads(std::shared_ptr<User> user)
    : UserComponent(user), mBitsConfigFetchToken(0), mHasFetchedBitsConfig(false) {
  Log(MessageLevel::Debug, "ctor: %s", user->GetUserName().c_str());
}

ttv::chat::ChatUserThreads::~ChatUserThreads() {
  Log(MessageLevel::Debug, "dtor");

  TTV_ASSERT(mState == State::Uninitialized);
}

void ttv::chat::ChatUserThreads::SetSettingRepository(std::shared_ptr<SettingRepository> settings) {
  mSettingRepository = settings;
}

void ttv::chat::ChatUserThreads::SetListener(std::shared_ptr<IChatUserThreadsListener> listener) {
  mListeners.ClearListeners();

  if (listener != nullptr) {
    mListeners.AddListener(listener);
  }
}

TTV_ErrorCode ttv::chat::ChatUserThreads::Initialize() {
  Log(MessageLevel::Debug, "ChatUserThreads::Initialize()");

  TTV_ASSERT(mTaskRunner != nullptr);

  TTV_ErrorCode ec = UserComponent::Initialize();
  if (TTV_FAILED(ec)) {
    return ec;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  mPubSub = user->GetComponentContainer()->GetComponent<PubSubClient>();
  if (mPubSub != nullptr) {
    mPubSubTopicListener = std::make_shared<PubSubTopicListener>(this);
    mPubSubTopicListenerHelper = std::make_shared<PubSubTopicListenerHelper>(mPubSub, mPubSubTopicListener);

    SubscribeTopics();
  }

  return TTV_EC_SUCCESS;
}

std::string ttv::chat::ChatUserThreads::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::ChatUserThreads::Shutdown() {
  TTV_ErrorCode ec = UserComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    if (mPubSubTopicListenerHelper != nullptr) {
      mPubSubTopicListenerHelper->Shutdown();
    }
  }

  if (mBitsConfigFetchToken != 0 && mBitsConfigRepository != nullptr) {
    mBitsConfigRepository->CancelFetch(mBitsConfigFetchToken);
  }

  return ec;
}

bool ttv::chat::ChatUserThreads::CheckShutdown() {
  if (!UserComponent::CheckShutdown()) {
    return false;
  }

  if (mBitsConfigFetchToken != 0) {
    return false;
  }

  // Make sure unsubscribed from pubsub topics
  if (mPubSubTopicListenerHelper != nullptr &&
      mPubSubTopicListenerHelper->GetState() != PubSubTopicListenerHelper::State::Shutdown) {
    return false;
  }

  return true;
}

void ttv::chat::ChatUserThreads::CompleteShutdown() {
  UserComponent::CompleteShutdown();

  mUser.reset();
  mPubSub.reset();
  mPubSubTopicListener.reset();
  mPubSubTopicListenerHelper.reset();
}

TTV_ErrorCode ttv::chat::ChatUserThreads::SubscribeTopics() {
  if (mPubSubTopicListenerHelper == nullptr) {
    return TTV_EC_FEATURE_DISABLED;
  }

  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  } else if (user->GetUserId() == 0) {
    Log(MessageLevel::Debug, "UserId not yet retrieved");

    return TTV_EC_USERINFO_NOT_AVAILABLE;
  } else if (mState != State::Initialized) {
    return TTV_EC_NOT_INITIALIZED;
  }

  // Generate the topic id we need to subscribe to
  char buffer[64];

  if (mWhisperPubSubTopic == "") {
    snprintf(buffer, sizeof(buffer), "whispers.%u", user->GetUserId());
    mWhisperPubSubTopic = buffer;
  }

  TTV_ErrorCode ec = TTV_EC_SUCCESS;

  if (!mPubSubTopicListenerHelper->ContainsTopic(mWhisperPubSubTopic)) {
    ec = mPubSubTopicListenerHelper->Subscribe(mWhisperPubSubTopic);
    TTV_ASSERT(TTV_SUCCEEDED(ec));
  }

  return ec;
}

void ttv::chat::ChatUserThreads::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  if (!mHasFetchedBitsConfig) {
    if (mFetchBitsConfigRetryTimer.CheckNextRetry() && mBitsConfigFetchToken == 0) {
      FetchBitsConfig();
    }
  }

  UserComponent::Update();
}

TTV_ErrorCode ttv::chat::ChatUserThreads::FetchBitsConfig() {
  if (mBitsConfigFetchToken != 0) {
    return TTV_EC_REQUEST_PENDING;
  }

  TTV_ErrorCode ec = mBitsConfigRepository->FetchGlobalBitsConfiguration(
    [this](TTV_ErrorCode callbackEc, const std::shared_ptr<BitsConfiguration>& config) {
      if (TTV_SUCCEEDED(callbackEc)) {
        mHasFetchedBitsConfig = true;
        mFetchBitsConfigRetryTimer.Clear();

        mBitsConfiguration = config;
      } else {
        mFetchBitsConfigRetryTimer.ScheduleNextRetry();
      }

      mBitsConfigFetchToken = 0;
    },
    mBitsConfigFetchToken);

  if (TTV_FAILED(ec)) {
    mFetchBitsConfigRetryTimer.ScheduleNextRetry();
  }

  return ec;
}

TTV_ErrorCode ttv::chat::ChatUserThreads::FetchEmoteSets() {
  auto user = mUser.lock();
  if (user == nullptr) {
    return TTV_EC_NEED_TO_LOGIN;
  }

  std::shared_ptr<UserEmoticonSets> emoticonSets = user->GetComponentContainer()->GetComponent<UserEmoticonSets>();

  if (emoticonSets == nullptr) {
    return TTV_EC_FEATURE_DISABLED;
  }

  TTV_ErrorCode ec = emoticonSets->FetchUserEmoticonSets(true, nullptr);

  return ec;
}

void ttv::chat::ChatUserThreads::OnUserInfoFetchComplete(TTV_ErrorCode ec) {
  if (TTV_SUCCEEDED(ec)) {
    SubscribeTopics();
  }

  UserComponent::OnUserInfoFetchComplete(ec);
}
void ttv::chat::ChatUserThreads::OnTopicSubscribeStateChanged(
  const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  if (topic == mWhisperPubSubTopic) {
    Log(MessageLevel::Debug, "ChatUserThreads::OnTopicSubscribeStateChanged: topic %s, state %s", topic.c_str(),
      PubSubClient::SubscribeState::ToString(state).c_str());
  }
}

void ttv::chat::ChatUserThreads::OnTopicMessageReceived(const std::string& topic, const json::Value& msg) {
  if (msg.isNull() || !msg.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mWhisperPubSubTopic) {
    const auto& jType = msg["type"];
    if (jType.isNull() || !jType.isString()) {
      Log(MessageLevel::Info, "No 'type' field, dropping message");
      return;
    }

    const auto& jData = msg["data_object"];
    if (jData.isNull() || !jData.isObject()) {
      Log(MessageLevel::Info, "No 'data' field, dropping message");
      return;
    }

    if (jType.asString() == "whisper_sent" || jType.asString() == "whisper_received") {
      std::vector<std::string> localUserNames;
      auto user = mUser.lock();
      if (user != nullptr) {
        localUserNames.emplace_back(user->GetUserName());
        localUserNames.emplace_back(user->GetDisplayName());
      }

      // Parse the message
      WhisperMessage whisperMessage;
      bool parsed = ParseMessageJson(jData, mTokenizationOptions, mBitsConfiguration, localUserNames, whisperMessage);
      if (!parsed) {
        Log(MessageLevel::Info, "Failed to parse thread message, dropping message");
        return;
      }

      // Parse the recipient
      const auto& jRecipient = jData["recipient"];
      if (jRecipient.isNull() || !jRecipient.isObject()) {
        Log(MessageLevel::Info, "No 'recipient' field, dropping message");
      }

      mListeners.Invoke([&user, &whisperMessage](std::shared_ptr<IChatUserThreadsListener> listener) {
        listener->ChatThreadRealtimeMessageReceived(user->GetUserId(), whisperMessage.threadId, whisperMessage);
      });
    }
  }
}

ttv::chat::ChatUserThreads::PubSubTopicListener::PubSubTopicListener(ChatUserThreads* owner) : mOwner(owner) {}

void ttv::chat::ChatUserThreads::PubSubTopicListener::OnTopicSubscribeStateChanged(
  PubSubClient* /*source*/, const std::string& topic, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode ec) {
  mOwner->OnTopicSubscribeStateChanged(topic, state, ec);
}

void ttv::chat::ChatUserThreads::PubSubTopicListener::OnTopicMessageReceived(
  PubSubClient* /*source*/, const std::string& topic, const json::Value& msg) {
  mOwner->OnTopicMessageReceived(topic, msg);
}

void ttv::chat::ChatUserThreads::PubSubTopicListener::OnTopicListenerRemoved(
  PubSubClient* /*source*/, const std::string& /*topic*/, TTV_ErrorCode /*ec*/) {
  // NOTE: The listener is only removed if we requested it to be removed or pubsub is shutting down
}
