/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/chatwriter.h"

ttv::chat::ChatWriter::ChatWriter() : mTransport(nullptr) {}

ttv::chat::ChatWriter::~ChatWriter() {}

void ttv::chat::ChatWriter::SetTransport(std::shared_ptr<IChatTransport> pTransport) {
  mTransport = pTransport;
}

void ttv::chat::ChatWriter::WriteRaw(const std::string& message) {
  std::string msgWithReturn = message;
  (void)msgWithReturn.append("\r\n");

  ttv::trace::Message(
    "ChatTransport", MessageLevel::Debug, "ChatWriter::WriteRaw: Send raw command: %s", msgWithReturn.c_str());

  TTV_ASSERT(mTransport);
  if (mTransport) {
    // TODO: Question to Drew. Would this not be better with two writes
    // instead of the append, as appending to a string can cause
    // an extra allocation?
    (void)mTransport->Write(msgWithReturn.c_str(), msgWithReturn.size());
  }
}

void ttv::chat::ChatWriter::WriteEvent(const ChatNetworkEvent& evt) {
  // the transport was not set so NOOP
  TTV_ASSERT(mTransport);
  if (mTransport == nullptr) {
    return;
  }

  const int idEvent = evt.GetEventID();

  std::string cmd = evt.GetEvent();
  if (!cmd.size()) {
    cmd = ChatNetworkEvent::EventIDToString(idEvent);
    TTV_ASSERT(cmd.size());
  }

  if (idEvent >= EVENT_OFFSET_COMMAND && idEvent <= IRC_CMD_UNKNOWN) {
    // <command> <parameter[0 to (n-1)]> :<parameter[n]>

    int nUsed = snprintf(mParseBuffer, sizeof(mParseBuffer), "%s", cmd.c_str());

    for (uint32_t i = 0; (i < evt.GetParamCount()) && (nUsed < static_cast<int>(sizeof(mParseBuffer)) - 1); ++i) {
      const std::string& sParam = evt.GetParam(i);
      TTV_ASSERT(sParam.c_str() != nullptr);

      if (i < evt.GetParamCount() - 1 || !evt.GetAutoPrefix()) {
        nUsed +=
          snprintf(mParseBuffer + nUsed, sizeof(mParseBuffer) - static_cast<size_t>(nUsed), " %s", sParam.c_str());
      } else {
        // prefix last parameter with ':'
        nUsed +=
          snprintf(mParseBuffer + nUsed, sizeof(mParseBuffer) - static_cast<size_t>(nUsed), " :%s", sParam.c_str());
      }
    }

    // If the line length is == POCKETIRC_MAX_IRC_LINE_LEN
    // then we need to add a null at the end, sntprintf won't
    mParseBuffer[nUsed] = '\0';

    WriteRaw(mParseBuffer);
  } else if ((idEvent >= EVENT_OFFSET_CTCP && idEvent <= IRC_CTCP_UNKNOWN) ||
             (idEvent >= EVENT_OFFSET_CTCP_REPLY && idEvent <= IRC_CTCP_RPL_UNKNOWN)) {
    if (evt.GetParamCount() < 1) {
      //_TRACE("ChatWriter(0x%08X)::WriteEvent() CTCP TO NOBODY NetworkEvent(idEvent = %d)", this, idEvent);
    } else {
      // PRIVMSG <parameter[0]> :\x01<CTCP command> <parameter[1 to n]\x01

      int nUsed = 0;
      if (idEvent >= EVENT_OFFSET_CTCP_REPLY && idEvent <= IRC_CTCP_RPL_UNKNOWN) {
        nUsed +=
          snprintf(mParseBuffer, sizeof(mParseBuffer), "NOTICE %s :\x01%s", evt.GetParam(0).c_str(), cmd.c_str());
      } else {
        nUsed +=
          snprintf(mParseBuffer, sizeof(mParseBuffer), "PRIVMSG %s :\x01%s", evt.GetParam(0).c_str(), cmd.c_str());
      }

      for (uint32_t i = 1; (i < evt.GetParamCount()) && (nUsed < static_cast<int>(sizeof(mParseBuffer)) - 2); ++i) {
        const std::string& sParam = evt.GetParam(i);
        TTV_ASSERT(sParam.c_str() != nullptr);

        nUsed +=
          snprintf(mParseBuffer + nUsed, sizeof(mParseBuffer) - static_cast<size_t>(nUsed), " %s", sParam.c_str());
      }

      // Add CTCP terminator, nUsed will be < POCKETIRC_MAX_IRC_LINE_LEN
      nUsed += snprintf(mParseBuffer + nUsed, sizeof(mParseBuffer) - static_cast<size_t>(nUsed), "\x01");

      // If the line length is == POCKETIRC_MAX_IRC_LINE_LEN
      // then we need to add a null at the end, sntprintf won't
      mParseBuffer[nUsed] = '\0';

      WriteRaw(mParseBuffer);
    }
  } else {
    //_TRACE("ChatWriter(0x%08X)::WriteEvent() GOT UNKNOWN NetworkEvent(idEvent = %d)", this, idEvent);
    TTV_ASSERT(false);
  }
}
