/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/ircstring.h"

#include "twitchsdk/core/stringutilities.h"

#include <cctype>

int ttv::chat::IrcStringGetWord(char* dst, const char* src, int maxLen, const char** ppEnd) {
  TTV_ASSERT(dst != nullptr);
  TTV_ASSERT(src != nullptr);
  TTV_ASSERT(maxLen >= 0);

  if (!dst || !src) {
    return 0;
  }

  int i = 0;
  for (; i < maxLen && src[i] != '\0' && !(src[i] == ' '); ++i) {
    dst[i] = src[i];
  }
  if (i < maxLen) {
    dst[i] = 0;
  }

  if (ppEnd) {
    *ppEnd = &src[i];
  }

  return i;
}

// I have seen non-breaking spaces used in EFNet channels, so we can't arbitrarily skip whitespace.
// IRC protocol specifies a single spaces as a parameter separator
const char* ttv::chat::IrcStringSkipSpaces(const char* psz) {
  const char* seek = psz;
  while (*seek != '\0' && *seek == ' ') {
    ++seek;
  }
  return seek;
}

bool ttv::chat::IsChannelString(const std::string& channel) {
  switch (channel[0]) {
    case '#':
    case '&':
    case '+':
    case '!':
      return true;
    default:
      return false;
  }
}

std::string ttv::chat::GetPrefixNick(const std::string& prefix) {
  if (prefix.size()) {
    std::string::size_type c = prefix.find_first_of('!');
    if (c != std::string::npos) {
      return prefix.substr(0, c);
    } else {
      return prefix;
    }
  }
  return "";
}

std::string ttv::chat::GetPrefixIdent(const std::string& prefix) {
  if (prefix.size()) {
    std::string::size_type start = prefix.find_first_of('!');
    if (start != std::string::npos) {
      ++start;

      std::string::size_type end = prefix.find_first_of('@', start);
      if (end != std::string::npos) {
        return prefix.substr(start, end - start);
      }
    }
  }
  return "";
}

std::string ttv::chat::GetPrefixHost(const std::string& prefix) {
  if (prefix.size()) {
    std::string::size_type start = prefix.find_first_of('!');
    if (start != std::string::npos) {
      start = prefix.find_first_of('@', start + 1);
      if (start != std::string::npos) {
        return prefix.substr(start + 1);
      }
    }
  }
  return "";
}

std::string ttv::chat::StripNickModes(const std::string& nick, const std::string& modeChars) {
  std::string::size_type start = nick.find_first_not_of(modeChars);
  if (start != std::string::npos) {
    return nick.substr(start);
  }
  return "";
}

std::string ttv::chat::StripNickModesAndInvalidChars(const std::string& nick, const std::string& modeChars) {
  std::string invalidChars = "<>,.()*!#$%&:;\"\'/?";
  std::string::size_type start = nick.find_first_not_of(modeChars + invalidChars);
  std::string::size_type end = nick.find_last_not_of(modeChars + invalidChars);

  if (start != std::string::npos) {
    end = end != std::string::npos ? end - start + 1 : std::string::npos;
    return nick.substr(start, end);
  }
  return "";
}

bool ttv::chat::NickHasMode(const std::string& nick, char mode) {
  return (nick.find_first_of(mode) != std::string::npos);
}

bool ttv::chat::IsTwitchChatUrl(const std::string& string) {
  // This algorithm is based off the following regex borrowed from web-client for determining if a token should be
  // URLified:
  // ^(?:(https?:\/\/)?((?:[\w#%\-+=:~]+\.)+[a-z]{2,8}(?:\/[\w./#%&()\-+=:?~]*)?))$
  //
  // We do not actually use std::regex, because that regex causes Microsoft's STL implementation to stack overflow when
  // building Release.
  const char* current = string.data();
  size_t stringLength = string.size();
  const char* stringEnd = current + stringLength;

  // Consume "http://" or "https://"
  if ((stringLength >= 8) && strncasecmp(current, "http", 4) == 0) {
    current += 4;

    if (*current == 's' || *current == 'S') {
      ++current;
    }

    if (strncmp(current, "://", 3) == 0) {
      current += 3;
    } else {
      current = string.data();
    }
  }

  if (*current == '.') {
    return false;
  }

  // Find the top-level domain.
  const char* lastPeriodPosition = nullptr;
  bool potentialValidTopLevelDomain = false;
  while (true) {
    if (current == stringEnd || *current == '/') {
      break;
    } else if (*current == ':' || *current == '%' || *current == '+' || *current == '~' || *current == '#' ||
               *current == '=' || *current == '-' || *current == '_') {
      potentialValidTopLevelDomain = false;
    } else if (*current == '.') {
      if (lastPeriodPosition != nullptr) {
        if (current - lastPeriodPosition == 1) {
          // Two periods back to back is not valid in a domain name.
          return false;
        }
      }

      lastPeriodPosition = current;
      potentialValidTopLevelDomain = true;
    } else if (!isalnum(static_cast<unsigned char>(*current))) {
      // Not a valid URL character. Bail out.
      return false;
    } else if (isdigit(static_cast<unsigned char>(*current)) && (current - lastPeriodPosition == 1)) {
      // if the next char after a '.' is a digit then invalid
      return false;
    }

    ++current;
  }

  size_t charactersSinceLastPeriod = static_cast<size_t>((current - lastPeriodPosition)) - 1;

  if (charactersSinceLastPeriod < 2 || charactersSinceLastPeriod > 8) {
    potentialValidTopLevelDomain = false;
  }

  if (!potentialValidTopLevelDomain) {
    return false;
  }

  // Found a top level domain. Make sure the rest of the string is valid URL characters.
  while (current < stringEnd) {
    if (*current != '/' && *current != '?' && *current != '&' && *current != ':' && *current != '%' &&
        *current != '+' && *current != '~' && *current != '#' && *current != '=' && *current != '_' &&
        *current != '-' && *current != '(' && *current != ')' && *current != '.' && *current != '@' &&
        !isalnum(static_cast<unsigned char>(*current))) {
      return false;
    }
    ++current;
  }
  return true;
}

bool ttv::chat::IsSimpleWordCharacter(char ch) {
  return (ch >= 'a' && ch <= 'z') || (ch >= 'A' && ch <= 'Z') || (ch >= '0' && ch <= '9') || ch == '_' || ch == '-';
}

std::string ttv::chat::GetWord(const std::string& str, uint32_t start, bool getRest, uint32_t& end) {
  // Find the start of the word
  while (start < str.size() && IsWhitespace(str[start])) {
    start++;
  }

  // No words
  if (start == str.size()) {
    end = static_cast<uint32_t>(str.size());
    return "";
  }
  // Read the rest of the string
  else if (getRest) {
    end = static_cast<uint32_t>(str.size());
    return str.substr(start);
  }
  // Find the end of the word
  else {
    end = start;
    while (end < str.size() && !IsWhitespace(str[end])) {
      end++;
    }

    return str.substr(start, end - start);
  }
}

std::string ttv::chat::ReplaceSubstring(const std::string& source, const std::string& target, const std::string& with) {
  // no target to replace
  if (target.length() == 0) {
    return source;
  }

  // nothing to match against
  if (source.length() == 0) {
    return source;
  }

  size_t index = 0;
  std::string result = source;

  for (;;) {
    index = result.find(target, index);
    if (index == std::string::npos) {
      break;
    }

    (void)result.replace(index, target.length(), with);
    index += with.length();
  }

  return result;
}

std::string ttv::chat::UnescapeMessageTag(const std::string& escapedMessageTag) {
  std::string unescapedMessageTag;

  const char* escapedStart = escapedMessageTag.c_str();
  size_t escapedSize = escapedMessageTag.size();

  size_t index = 0;
  while (escapedStart[index] != '\0') {
    size_t backslashIndex = escapedMessageTag.find("\\", index);

    if (backslashIndex == std::string::npos) {
      // Append the remainder of the string, then break out of the loop.
      unescapedMessageTag.append(escapedStart + index, escapedSize - index);
      break;
    }

    // Append the string leading up to the backslash
    unescapedMessageTag.append(escapedStart + index, backslashIndex - index);
    switch (escapedStart[backslashIndex + 1]) {
      case ':': {
        // Semicolon
        unescapedMessageTag.append(";");
        index = backslashIndex + 2;
        break;
      }

      case 's': {
        // Space
        unescapedMessageTag.append(" ");
        index = backslashIndex + 2;
        break;
      }

      case '\\': {
        // Explicit backslash
        unescapedMessageTag.append("\\");
        index = backslashIndex + 2;
        break;
      }

      case 'r': {
        // Carriage return
        unescapedMessageTag.append("\r");
        index = backslashIndex + 2;
        break;
      }

      case 'n': {
        // Line feed
        unescapedMessageTag.append("\n");
        index = backslashIndex + 2;
        break;
      }

      default: {
        // Some other character, which is unrecognized (and technically unsupported).
        // In this case, just recover by adding the backslash directly and continuing the parse.
        unescapedMessageTag.append("\\");
        index = backslashIndex + 1;

        break;
      }
    }
  }

  return unescapedMessageTag;
}

bool ttv::chat::Compare(const std::string& one, const std::string& two, bool caseSensitive) {
  return caseSensitive ? (one == two) : (strcasecmp(one.c_str(), two.c_str()) == 0);
}
