/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/multiviewnotifications.h"

#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"

namespace {
const char* kLoggerName = "MultiviewNotifications";
const char* kTopicPrefix = "multiview-chanlet-update.";
}  // namespace

ttv::chat::MultiviewNotifications::MultiviewNotifications(const std::shared_ptr<User>& user, ChannelId channelId)
    : PubSubComponent(user), mPubSubTopic(kTopicPrefix + std::to_string(channelId)), mChannelId(channelId) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::MultiviewNotifications::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::MultiviewNotifications::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::MultiviewNotifications::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "SquadNotificationsListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::MultiviewNotifications::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) {
  if (jVal.isNull() || !jVal.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    const auto& jType = jVal["type"];
    if (jType.isNull() || !jType.isString() || jType.asString() != "chanlet") {
      ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message unknown type, dropping");
      return;
    }

    const auto& jChanlet = jVal["chanlet"];
    if (!jChanlet.isNull() && jChanlet.isObject()) {
      Chanlet chanlet;
      if (!ttv::json::ToObject(jChanlet, chanlet)) {
        return;
      }

      auto user = mUser.lock();
      if (user != nullptr) {
        mListener->ChanletUpdated(user->GetUserId(), mChannelId, std::move(chanlet));
      }
    } else {
      ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message missing chanlet info");
      return;
    }
  }
}
