/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/squadnotifications.h"

#include "twitchsdk/core/json/jsonobjectdescriptions.h"

namespace {
const char* kLoggerName = "SquadNotifications";
const char* kTopicPrefix = "squad-updates.";
}  // namespace

ttv::chat::SquadNotifications::SquadNotifications(const std::shared_ptr<User>& user, const std::string& squadId)
    : PubSubComponent(user), mPubSubTopic(kTopicPrefix + squadId) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::SquadNotifications::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::SquadNotifications::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::SquadNotifications::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "SquadNotificationsListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::SquadNotifications::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) {
  if (jVal.isNull() || !jVal.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    const auto& jType = jVal["type"];
    if (jType.isNull() || !jType.isString() || jType.asString() != "squad") {
      ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message unknown type, dropping");
      return;
    }

    const auto& jSquad = jVal["squad"];
    if (!jSquad.isNull() && jSquad.isObject()) {
      SquadInfo squad;
      if (!ttv::json::ToObject(jSquad, squad)) {
        return;
      }

      if (squad.status == SquadStatus::Ended) {
        mListener->SquadEnded();
      } else {
        mListener->SquadUpdated(std::move(squad));
      }
    } else {
      ttv::trace::Message(kLoggerName, MessageLevel::Error, "Pub sub message missing squad info");
      return;
    }
  }
}
