/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/subscriptionsnotifications.h"

#include "twitchsdk/core/json/corejsonutil.h"

namespace {
const char* kLoggerName = "SubscriptionsNotifications";
const char* kTopicPrefix = "user-subscribe-events-v1.";
}  // namespace

ttv::chat::SubscriptionsNotifications::SubscriptionsNotifications(const std::shared_ptr<User>& user)
    : PubSubComponent(user), mPubSubTopic(kTopicPrefix + std::to_string(user->GetUserId())) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::SubscriptionsNotifications::GetLoggerName() const {
  return kLoggerName;
}

TTV_ErrorCode ttv::chat::SubscriptionsNotifications::Dispose() {
  if (mDisposerFunc != nullptr) {
    mDisposerFunc();

    mDisposerFunc = nullptr;
  }

  return TTV_EC_SUCCESS;
}

void ttv::chat::SubscriptionsNotifications::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLoggerName, MessageLevel::Debug, "SubscriptionsNotificationsListener SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));
}

void ttv::chat::SubscriptionsNotifications::OnTopicMessageReceived(
  const std::string& topic, const ttv::json::Value& jVal) {
  if (jVal.isNull() || !jVal.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    UserId userId;
    ChannelId channelId;

    if (!ParseUserId(jVal["user_id"], userId)) {
      Log(MessageLevel::Error, "Could not parse user ID, dropping");
      return;
    }

    auto user = mUser.lock();
    if (user == nullptr || user->GetUserId() == 0 || userId != user->GetUserId()) {
      Log(MessageLevel::Error, "Ivnalid user id, dropping");
      return;
    }

    if (!ParseChannelId(jVal["channel_id"], channelId)) {
      Log(MessageLevel::Error, "Could not parse channel ID, dropping");
      return;
    }

    if (mListener != nullptr) {
      mListener->SubscribedToChannel(userId, channelId);
    }
  }
}
