/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chataddnewroomtask.h"

#include "twitchsdk/chat/internal/graphql/generated/createroomqueryinfo.h"
#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($input: CreateRoomInput!) {createRoom(input:$input) { room
// { id name topic rolePermissions { read send } self { lastReadAt isMuted isArchived isUnread unreadMentionCount }
// owner { id login displayName description profileImageURL(width: 300) createdAt} } } }", "variables": { "input" : {
// "name": "$ROOMNAME", "topic": "$TOPIC", "minimumReadMessagesRole": "$MINIMUMREADROLE", "minimumSendMessagesRole":
// "$MINIUMSENDROLE" }}
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatAddNewRoomTask::ChatAddNewRoomTask(const std::string& roomName, const std::string& topic,
  RoomRolePermissions roomPermissions, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mRoomName(roomName),
      mRoomTopic(topic),
      mRoomPermissions(roomPermissions) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(roomName.size() > 0);
  TTV_ASSERT(topic.size() > 0);
  TTV_ASSERT(mRoomPermissions.read != RoomRole::Unknown);
  TTV_ASSERT(mRoomPermissions.send != RoomRole::Unknown);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatAddNewRoomTask created");
}

void ttv::chat::ChatAddNewRoomTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  ttv::json::Value root;

  root["query"] = graphql::GetCreateRoomMutation();
  root["variables"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["input"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["input"]["name"] = mRoomName;
  root["variables"]["input"]["topic"] = mRoomTopic;
  ToJson(mRoomPermissions.read, root["variables"]["input"]["minimumReadMessagesRole"]);
  ToJson(mRoomPermissions.send, root["variables"]["input"]["minimuimSendMessagesRole"]);

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatAddNewRoomTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    ttv::json::Value jsonVal;
    ttv::json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatAddNewRoomTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    if (!ParseCreateRoom(jsonVal, mResult, mError)) {
      if (mError.code == GraphQLErrorCode::SUCCESS) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatAddNewRoomTask::ProcessResponse - Parsing json values failed");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      } else {
        ttv::trace::Message(
          GetTaskName(), MessageLevel::Error, "Inside ChatAddNewRoomTask::ProcessResponse - GraphQL Error received");
        mTaskStatus = TTV_EC_GRAPHQL_ERROR;
      }
    }
  } else {
    ttv::trace::Message("ChatAddNewRoomTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatAddNewRoomTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mError), std::move(mResult));
  }
}
