/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatapitask.h"

#include "twitchsdk/core/httprequest.h"
#include "twitchsdk/core/httprequestutils.h"

namespace {
using namespace ttv;

const uint kDefaultRequestTimeoutInSec = 10;
}  // namespace

//-----------------------------------------------------------------------
ttv::chat::ChatApiTask::HttpRequestInfo::HttpRequestInfo()
    : httpReqType(HTTP_INVALID_REQUEST), timeOutInSecs(kDefaultRequestTimeoutInSec), numRetries(0) {}

//-----------------------------------------------------------------------
ttv::chat::ChatApiTask::~ChatApiTask() {}

//-----------------------------------------------------------------------
void ttv::chat::ChatApiTask::Run() {
  if (!mAborted) {
    HttpRequestInfo requestInfo;
    FillHttpRequestInfo(requestInfo);

    // Add the OAuth token if configured
    if (mAuthToken.size() > 0) {
      if (!ContainsHttpParameter(requestInfo.requestHeaders, "Authorization")) {
        requestInfo.requestHeaders.emplace_back(HttpParam("Authorization", "OAuth " + mAuthToken));
      }
    }

    // Prefer to receive gzipped responses if no other specified
    if (!ContainsHttpParameter(requestInfo.requestHeaders, "Accept-Encoding")) {
      requestInfo.requestHeaders.emplace_back(HttpParam("Accept-Encoding", "gzip"));
    }

    std::function<bool(uint, const std::map<std::string, std::string>&, void*)> headersCallback =
      [this](uint statusCode, const std::map<std::string, std::string>& headers, void* userData) {
        return HeadersCallback(statusCode, headers, userData);
      };

    std::function<void(uint, const std::vector<char>&, void*)> responseCallback =
      [this](uint statusCode, const std::vector<char>& response, void* userData) {
        return ResponseCallback(statusCode, response, userData);
      };

    using namespace std::placeholders;
    TTV_ErrorCode ec = SendHttpRequest(GetTaskName(), requestInfo.url, requestInfo.requestHeaders,
      reinterpret_cast<const uint8_t*>(requestInfo.requestBody.c_str()), requestInfo.requestBody.size(),
      requestInfo.httpReqType, requestInfo.timeOutInSecs, requestInfo.numRetries, headersCallback, responseCallback,
      nullptr);

    // if the request attempt itself failed then return that error
    if (TTV_FAILED(ec)) {
      mTaskStatus = ec;
    }
  } else {
    mTaskStatus = TTV_EC_REQUEST_ABORTED;
  }
}

//-----------------------------------------------------------------------
void ttv::chat::ChatApiTask::OnComplete() {
  if (mTaskCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mTaskCallback(mTaskStatus, mUserData);
  }
}

//-----------------------------------------------------------------------
bool ttv::chat::ChatApiTask::ProcessHeaders(const std::map<std::string, std::string>& /*headers*/) {
  return true;
}

//-----------------------------------------------------------------------
bool ttv::chat::ChatApiTask::HeadersCallback(
  uint /*statusCode*/, const std::map<std::string, std::string>& headers, void* /*userData*/) {
  return ProcessHeaders(headers);
}

//-----------------------------------------------------------------------
void ttv::chat::ChatApiTask::ResponseCallback(uint statusCode, const std::vector<char>& response, void* /*userData*/) {
  if (!mAborted) {
    if (statusCode >= 200 && statusCode < 300) {
      // Set the status to SUCCESS here; the task can override this when processing the response
      mTaskStatus = TTV_EC_SUCCESS;
      ProcessResponse(response);
    } else {
      std::string msg(response.data(), response.size());
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "HTTP request failed with status code %d. Message: %s",
        statusCode, msg.c_str());
      mTaskStatus = TTV_EC_API_REQUEST_FAILED;
    }
  }
}
