/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatbanusertask.h"

#include "twitchsdk/chat/internal/graphql/generated/banuserqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($input: BanUserFromChatRoomInput!)
// {banUserFromChatRoom(input:$input) { ban { createdAt expiresAt isPermanent } }", "variables": { "input" : {
// "channelID": "$CHANNELID", "bannedUserLogin": "$BANNEDLOGIN", "expriesIn": "$EXPIRESIN" }}
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatBanUserTask::ChatBanUserTask(
  ChannelId channelId, const std::string& bannedUserName, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mBannedUserName(bannedUserName),
      mChannelId(channelId),
      mDurationSeconds(0) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(bannedUserName.size() > 0);
  TTV_ASSERT(mChannelId != 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatBanUserTask created");
}

void ttv::chat::ChatBanUserTask::SetTimeout(uint32_t durationSeconds) {
  mDurationSeconds = durationSeconds;
}

void ttv::chat::ChatBanUserTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = graphql::GetBanUserMutation();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["input"] = json::Value(json::objectValue);
  root["variables"]["input"]["channelID"] = std::to_string(mChannelId);
  root["variables"]["input"]["bannedUserLogin"] = mBannedUserName;

  if (mDurationSeconds != 0) {
    root["variables"]["input"]["expiresIn"] = std::to_string(mDurationSeconds) + "s";
  }

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatBanUserTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatBanUserTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    if (!ParseBanUser(jsonVal, mError)) {
      if (mError.code == GraphQLErrorCode::SUCCESS) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatRoomSendMessageTask::ProcessResponse - Parsing json values failed");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      } else {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatRoomSendMessageTask::ProcessResponse - GraphQL Error received");
        mTaskStatus = TTV_EC_GRAPHQL_ERROR;
      }
    }
  } else {
    ttv::trace::Message("ChatBanUserTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatBanUserTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mError));
  }
}
