/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatchangeuserblocktask.h"

#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"

#include <sstream>

// https://github.com/justintv/Twitch-API/blob/master/v3_resources/blocks.md

namespace {
const char* kApiVersion = "application/vnd.twitchtv.v5+json";
}

// Example curls
// curl -X PUT -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H "Accept: application/vnd.twitchtv.v5+json"
// -H "Content-Length: 0" "https://api.twitch.tv/kraken/users/$USERID1/blocks/$USERID2" | python -m json.tool PUT
// /v5/users/:user_id/blocks/:target_id curl -X DELETE -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H
// "Accept: application/vnd.twitchtv.v5+json" -H "Content-Length: 0"
// "https://api.twitch.tv/kraken/users/$USERID1/blocks/$USERID2" | python -m json.tool DELETE
// /v5/users/:user_id/blocks/:target_id

//-----------------------------------------------------------------------
ttv::chat::ChatChangeUserBlockTask::ChatChangeUserBlockTask(UserId userId, const std::string& authToken,
  UserId blockUserId, bool block, const std::string& reason, Callback callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(callback),
      mReason(reason),
      mUserId(userId),
      mBlockUserId(blockUserId),
      mBlock(block),
      mWhisper(false) {
  TTV_ASSERT(userId != 0);
  TTV_ASSERT(blockUserId != 0);
  TTV_ASSERT(callback);

  ttv::trace::Message("ChatChangeUserBlockTask", MessageLevel::Info, "ChatChangeUserBlockTask created");
}

//-----------------------------------------------------------------------
void ttv::chat::ChatChangeUserBlockTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << "https://api.twitch.tv/kraken/users/" << mUserId << "/blocks/" << mBlockUserId;

  Uri url(ss.str());

  if (mBlock) {
    if (mReason != "") {
      url.SetParam("reason", mReason);
    }

    url.SetParam("whisper", mWhisper);
  }

  requestInfo.url = url;

  requestInfo.requestHeaders.emplace_back(HttpParam("Accept", kApiVersion));

  requestInfo.httpReqType = mBlock ? HTTP_PUT_REQUEST : HTTP_DELETE_REQUEST;
}

//-----------------------------------------------------------------------
void ttv::chat::ChatChangeUserBlockTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mUserId, mBlockUserId, mBlock);
  }
}
