/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatchanneluserstask.h"

#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/stringutilities.h"

#include <sstream>

namespace {
const char* kApiVersion = "application/vnd.twitchtv.v3+json";
#define CHANNEL_NAME_TEMPLATE "{channel_name}"
const char* kChannelNameTemplate = CHANNEL_NAME_TEMPLATE;
const char* kChattersURL = "https://tmi.twitch.tv/group/user/" CHANNEL_NAME_TEMPLATE "/chatters";
}  // namespace

ttv::chat::ChatChannelUsersTask::Result::Result() : cacheControlMaxAgeSeconds(0) {}

ttv::chat::ChatChannelUsersTask::ChatChannelUsersTask(const std::string& channelName, Callback callback)
    : HttpTask(nullptr, nullptr, nullptr), mCallback(callback), mChannelName(channelName) {
  TTV_ASSERT(IsValidChannelName(channelName));

  ttv::trace::Message("ChatChannelUsersTask", MessageLevel::Info, "ChatChannelUsersTask created");
}

void ttv::chat::ChatChannelUsersTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kChattersURL;
  (void)requestInfo.url.replace(requestInfo.url.find(kChannelNameTemplate), strlen(kChannelNameTemplate), mChannelName);
  requestInfo.httpReqType = HTTP_GET_REQUEST;
  requestInfo.requestHeaders.emplace_back(HttpParam("Accept", kApiVersion));
}

bool ttv::chat::ChatChannelUsersTask::ProcessHeaders(
  uint /*statusCode*/, const std::map<std::string, std::string>& headers) {
  mResult = std::make_shared<Result>();

  // Determine the max-age of the list so we know how often we should be refreshing it
  auto iter = headers.find("Cache-Control");
  if (iter == headers.end()) {
    return true;
  }

  std::string cacheControl = iter->second;
  std::vector<std::string> tokens;
  ttv::Split(cacheControl, tokens, ',', false);

  const std::string kMaxAge("max-age");

  for (auto token : tokens) {
    // Trim leading and trailing whitespace
    Trim(token);

    // Lower case
    (void)std::transform(token.begin(), token.end(), token.begin(), ::tolower);

    if (token.substr(0, kMaxAge.size()) == kMaxAge) {
      std::vector<std::string> kvp;
      ttv::Split(token, kvp, '=', false);

      if (kvp.size() == 2) {
        token = kvp[1];
        Trim(token);

        ParseNum(token, mResult->cacheControlMaxAgeSeconds);
      }

      break;
    }
  }

  return true;
}

void ttv::chat::ChatChannelUsersTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message("ChatChannelUsersTask", MessageLevel::Error,
        "Inside ChatChannelUsersTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    const auto& jChatters = jsonVal["chatters"];
    if (jChatters.isNull() || !jChatters.isObject()) {
      ttv::trace::Message("ChatChannelUsersTask", MessageLevel::Error,
        "Inside ChatChannelUsersTask::ProcessResponse - Invalid chatters element");
      return;
    }

    auto jGroup = jChatters["moderators"];
    if (!jGroup.isNull() && jGroup.isArray()) {
      ParseUserList(jGroup, mResult->users.moderators);
    }

    jGroup = jChatters["global_mods"];
    if (!jGroup.isNull() && jGroup.isArray()) {
      ParseUserList(jGroup, mResult->users.globalModerators);
    }

    jGroup = jChatters["staff"];
    if (!jGroup.isNull() && jGroup.isArray()) {
      ParseUserList(jGroup, mResult->users.staff);
    }

    jGroup = jChatters["admins"];
    if (!jGroup.isNull() && jGroup.isArray()) {
      ParseUserList(jGroup, mResult->users.admins);
    }

    jGroup = jChatters["vips"];
    if (!jGroup.isNull() && jGroup.isArray()) {
      ParseUserList(jGroup, mResult->users.vips);
    }

    jGroup = jChatters["viewers"];
    if (!jGroup.isNull() && jGroup.isArray()) {
      ParseUserList(jGroup, mResult->users.viewers);
    }

    const auto& jChatterCount = jsonVal["chatter_count"];
    if (!jChatterCount.isNull() && jChatterCount.isNumeric()) {
      mResult->users.totalUserCount = static_cast<uint32_t>(jChatterCount.asUInt());
    }
  } else {
    mResult.reset();

    ttv::trace::Message("ChatChannelUsersTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatChannelUsersTask::ParseUserList(ttv::json::Value jGroup, std::vector<std::string>& users) {
  for (const auto& jUsername : jGroup) {
    if (!jUsername.isString()) {
      continue;
    }

    users.push_back(jUsername.asString());
  }
}

void ttv::chat::ChatChannelUsersTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
