/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatfetchchannelmoderatorstask.h"

#include "twitchsdk/chat/internal/graphql/generated/fetchchannelmoderatorsqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "query($channelId: ID!) {user(id:$channelId) { mods(first: 100) {
// edges { node { displayName } } } } }", "variables": { "channelId" : "$CHANNELID" }
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatFetchChannelModeratorsTask::ChatFetchChannelModeratorsTask(
  ChannelId channelId, const std::string& cursor, Callback&& callback)
    : HttpTask(nullptr, nullptr, nullptr), mCursor(cursor), mCallback(std::move(callback)), mChannelId(channelId) {
  TTV_ASSERT(mChannelId != 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatFetchChannelModeratorsTask created");
}

void ttv::chat::ChatFetchChannelModeratorsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = graphql::GetFetchChannelModeratorsQuery();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["channelId"] = std::to_string(mChannelId);
  root["variables"]["cursor"] = mCursor;

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatFetchChannelModeratorsTask::ProcessResponse(
  uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatFetchChannelModeratorsTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    if (!ParseGraphQLUserMods(jsonVal, mResult, mResultNextCursor)) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatFetchChannelModeratorsTask::ProcessResponse - Parsing json values failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatFetchChannelModeratorsTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatFetchChannelModeratorsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mResult), std::move(mResultNextCursor));
  }
}
