/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatfetchchannelroomstask.h"

#include "twitchsdk/chat/internal/graphql/generated/fetchchannelroomsqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "query($roomId: ID!, $cursor: Cursor) {room(id:$roomId) { messages
// (after: $cursor){ edges { cursor node { id content { fragments { text content { ... on Emote { id } __typename } } }
// } } pageInfo { hasNextPage hasPreviousPage } } } }", "variables": { "roomId": "$ROOMID$, "cursor":"$CURSOR" }
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatFetchChannelRoomsTask::ChatFetchChannelRoomsTask(
  ChannelId channelId, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(std::move(callback)), mChannelId(channelId) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(channelId != 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatFetchChannelRoomsTask created");
}

void ttv::chat::ChatFetchChannelRoomsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = graphql::GetFetchChannelRoomsQuery();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["channelId"] = std::to_string(mChannelId);

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatFetchChannelRoomsTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatFetchChannelRoomsTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    if (!ParseGraphQLChatRooms(jsonVal, mResult)) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatFetchChannelRoomsTask::ProcessResponse - Parsing json values failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatFetchChannelRoomsTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatFetchChannelRoomsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
