/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatgetbadgestask.h"

#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

#include <sstream>

namespace {
const char* kHostName = "https://badges.twitch.tv";
}

// Global badges:
//   curl -X GET -H "Client-Id: $CLIENTID" "https://badges.twitch.tv/v1/badges/global/display?language=EN" | python -m
//   json.tool
//
// Channel badges for user:
//   curl -X GET -H "Client-Id: $CLIENTID" "https://badges.twitch.tv/v1/badges/channels/$CHANNELID/display?language=EN"
//   | python -m json.tool

ttv::chat::ChatGetBadgesTask::ChatGetBadgesTask(ChannelId channelId, Callback&& callback)
    : HttpTask(nullptr, nullptr, nullptr), mCallback(std::move(callback)), mChannelId(channelId), mLanguage("EN") {
  if (mChannelId == 0) {
    ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetBadgesTask created for global badges");
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetBadgesTask created for channel badges");
  }
}

void ttv::chat::ChatGetBadgesTask::SetLanguage(const std::string& language) {
  mLanguage = language;
}

void ttv::chat::ChatGetBadgesTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << kHostName << "/v1/badges/";

  // Global badges
  if (mChannelId == 0) {
    ss << "global/";
  }
  // Channel badges
  else {
    ss << "channels/" << mChannelId << "/";
  }

  ss << "display";

  Uri url(ss.str());

  if (mLanguage != "") {
    url.SetParam("language", mLanguage);
  }

  requestInfo.url = url.GetUrl();
  requestInfo.httpReqType = HTTP_GET_REQUEST;
}

void ttv::chat::ChatGetBadgesTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatGetBadgesTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    const auto& jBadgeSets = jsonVal["badge_sets"];
    if (jBadgeSets.isNull() || !jBadgeSets.isObject()) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatGetBadgesTask::ProcessResponse - Invalid 'badge_sets' element");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    std::map<std::string, Badge> map;

    for (auto iter = jBadgeSets.begin(); iter != jBadgeSets.end(); iter++) {
      Badge badge;
      parsed = ParseBadge(iter, badge);

      if (parsed) {
        map[badge.name] = std::move(badge);
      } else {
        ttv::trace::Message(
          GetTaskName(), MessageLevel::Error, "Inside ChatGetBadgesTask::ProcessResponse - Invalid badge data");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        return;
      }
    }

    mResult.language = mLanguage;
    mResult.badges = std::move(map);
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

bool ttv::chat::ChatGetBadgesTask::ParseBadgeVersion(
  const ttv::json::ValueConstIterator& jIter, BadgeVersion& version) {
  const auto jName = jIter.key();
  const auto& jVersion = (*jIter);

  if (!jName.isString() || jVersion.isNull() || !jVersion.isObject()) {
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return false;
  }

  const auto& jTitle = jVersion["title"];
  if (!jTitle.isString()) {
    ttv::trace::Message(
      GetTaskName(), MessageLevel::Error, "Inside ChatGetBadgesTask::ParseBadgeVersion - Invalid 'title' element");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return false;
  }

  const auto& jDescription = jVersion["description"];
  if (!jDescription.isString()) {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error,
      "Inside ChatGetBadgesTask::ParseBadgeVersion - Invalid 'description' element");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return false;
  }

  version.name = jName.asString();
  version.title = jTitle.asString();
  version.description = jDescription.asString();

  const auto& jClickAction = jVersion["click_action"];
  const auto& jClickUrl = jVersion["click_url"];

  if (jClickAction.isString()) {
    std::string action = jClickAction.asString();

    if (action == "subscribe_to_channel") {
      version.clickAction = BadgeVersion::Action::Subscribe;
    } else if (action == "visit_url") {
      version.clickAction = BadgeVersion::Action::VisitUrl;
    } else if (action == "turbo") {
      version.clickAction = BadgeVersion::Action::GetTurbo;
    } else if (action == "none") {
      version.clickAction = BadgeVersion::Action::None;
    } else {
      // Handle new action types
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatGetBadgesTask::ParseBadgeVersion - Unhandled action type: %s", action.c_str());
    }
  } else {
    version.clickAction = BadgeVersion::Action::None;
  }

  if (jClickUrl.isString()) {
    version.clickUrl = jClickUrl.asString();
  }

  const std::string kImageScalePrefix("image_url_");
  const std::string kImageScaleSuffix("x");

  std::vector<BadgeImage> images;

  // Pull out image scales
  for (auto iter = jVersion.begin(); iter != jVersion.end(); iter++) {
    const auto jKey = iter.key();
    if (!jKey.isString()) {
      continue;
    }

    std::string scale = jKey.asString();

    // Look for keys of the form "image_url_1x"

    if (!StartsWith(scale, kImageScalePrefix) || !EndsWith(scale, kImageScaleSuffix)) {
      continue;
    }

    const auto& jUrl = (*iter);
    if (!jUrl.isString()) {
      continue;
    }

    // Chop off the unimportant values
    scale = scale.substr(0, scale.size() - kImageScaleSuffix.size());
    scale = scale.substr(kImageScalePrefix.size());

    BadgeImage image;

    if (!ParseNum(scale, image.scale)) {
      continue;
    }

    image.url = jUrl.asString();

    images.push_back(image);
  }

  version.images = std::move(images);

  return true;
}

bool ttv::chat::ChatGetBadgesTask::ParseBadge(const ttv::json::ValueConstIterator& jIter, Badge& badge) {
  const auto jName = jIter.key();
  const auto& jBadge = (*jIter);

  if (!jName.isString() || jBadge.isNull() || !jBadge.isObject()) {
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return false;
  }

  const auto& jVersions = jBadge["versions"];
  if (jVersions.isNull() || !jVersions.isObject()) {
    ttv::trace::Message(
      GetTaskName(), MessageLevel::Error, "Inside ChatGetBadgesTask::ParseBadge - Invalid 'versions' element");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return false;
  }

  std::map<std::string, BadgeVersion> versions;

  for (auto iter = jVersions.begin(); iter != jVersions.end(); iter++) {
    BadgeVersion version;
    bool parsed = ParseBadgeVersion(iter, version);

    if (parsed) {
      versions[version.name] = std::move(version);
    } else {
      continue;
    }
  }

  badge.name = jName.asString();
  badge.versions = std::move(versions);

  return true;
}

void ttv::chat::ChatGetBadgesTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
