/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatgetbitsconfigtask.h"

#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/stringutilities.h"

#include <sstream>

namespace {
// const char* kBitsConfigUrl = "https://api.twitch.tv/v5/bits/actions";
const char* kBitsConfigUrl = "";  // This request has been deprecated, as it is only used for animated cheermotes.
}  // namespace

// curl -X GET -H "Client-Id: $CLIENTID" "https://api.twitch.tv/v5/bits/actions?" | python -m json.tool
// GET /v5/bits/actions

ttv::chat::ChatGetBitsConfigTask::ChatGetBitsConfigTask(ChannelId channelId, Callback callback)
    : HttpTask(nullptr, nullptr, nullptr), mCallback(callback), mChannelId(channelId) {
  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetBitsConfigTask with ChannelId created");
}

void ttv::chat::ChatGetBitsConfigTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << kBitsConfigUrl;

  if (mChannelId > 0) {
    ss << "?channel_id=" << mChannelId;
  }

  Uri url(ss.str());
  requestInfo.url = url.GetUrl();
  // SDK-778 Visage v5 API
  requestInfo.httpReqType = HTTP_GET_REQUEST;
}

void ttv::chat::ChatGetBitsConfigTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
        "Inside ChatGetBitsConfigTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    std::vector<BitsConfiguration::Cheermote> cheermotes;
    const auto& jActions = jsonVal["actions"];
    if (jActions.isNull() || !jActions.isArray()) {
      ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
        "Inside ChatGetBitsConfigTask::ProcessResponse - No JSON bits element found");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    } else {
      for (const auto& jAction : jActions) {
        BitsConfiguration::Cheermote action;
        const auto& jPrefix = jAction["prefix"];
        if (jPrefix.isNull() || !jPrefix.isString()) {
          ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
            "Inside ChatGetBitsConfigTask::ProcessResponse - No action prefix found");
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
          return;
        }

        action.prefix = jPrefix.asString();

        const auto& jTiers = jAction["tiers"];
        if (jTiers.isNull() || !jTiers.isArray()) {
          ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
            "Inside ChatGetBitsConfigTask::ProcessResponse - No JSON tiers element found");
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
          return;
        } else {
          for (const auto& jTier : jTiers) {
            if (jTier.isNull() || !jTier.isObject()) {
              ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
                "Inside ChatGetBitsConfigTask::ProcessResponse - Tier is not an object, skipping it");
              continue;
            }

            const auto& jMinBits = jTier["min_bits"];
            const auto& jColor = jTier["color"];
            const auto& jImages = jTier["images"];
            const auto& jCanCheer = jTier["can_cheer"];

            if (jMinBits.isNull() || !jMinBits.isNumeric() || jColor.isNull() || !jColor.isString() ||
                jImages.isNull() || !jImages.isObject() || jCanCheer.isNull() || !jCanCheer.isBool()) {
              ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
                "Inside ChatGetBitsConfigTask::ProcessResponse - Tier missing min_bits, can_cheer, or color, skipping it");
              continue;
            }

            BitsConfiguration::CheermoteTier tier;
            tier.bits = static_cast<uint32_t>(jMinBits.asUInt());
            tier.tierID = std::to_string(tier.bits);
            ParseColor(jColor.asString(), tier.color);
            tier.canCheer = jCanCheer.asBool();
            tier.canShowInBitsCard = true;

            for (auto backgroundIter = jImages.begin(); backgroundIter != jImages.end(); backgroundIter++) {
              auto jBackground = backgroundIter.key();
              if (jBackground.isNull() || !jBackground.isString()) {
                ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
                  "Inside ChatGetBitsConfigTask::ProcessResponse - Invalid background value");
                continue;
              }

              auto jTierAnimated = *backgroundIter;
              for (auto animatedIter = jTierAnimated.begin(); animatedIter != jTierAnimated.end(); animatedIter++) {
                auto jAnimated = animatedIter.key();
                if (jAnimated.isNull() || !jAnimated.isString()) {
                  ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
                    "Inside ChatGetBitsConfigTask::ProcessResponse - Invalid animation state value");
                  continue;
                }

                auto jTierScales = *animatedIter;
                for (auto scalesIter = jTierScales.begin(); scalesIter != jTierScales.end(); scalesIter++) {
                  auto jScale = scalesIter.key();
                  float scale;
                  if (!ParseFloat(jScale, scale)) {
                    ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
                      "Inside ChatGetBitsConfigTask::ProcessResponse - Invalid animation state value");
                    continue;
                  }

                  auto jUrl = *scalesIter;
                  if (jUrl.isNull() || !jUrl.isString()) {
                    ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error,
                      "Inside ChatGetBitsConfigTask::ProcessResponse - Invalid url value");
                    continue;
                  }

                  tier.images.emplace_back();

                  auto& image = tier.images.back();
                  image.url = jUrl.asString();

                  if (jBackground.asString() == "light") {
                    image.theme = BitsConfiguration::CheermoteImage::Theme::Light;
                  } else if (jBackground.asString() == "dark") {
                    image.theme = BitsConfiguration::CheermoteImage::Theme::Dark;
                  } else {
                    image.theme = BitsConfiguration::CheermoteImage::Theme::Unknown;
                  }

                  image.isAnimated = (jAnimated.asString() == "animated");
                  image.dpiScale = scale;
                }
              }
            }
            action.tiers.push_back(std::move(tier));
          }
        }

        BitsConfiguration::Cheermote::Type type = BitsConfiguration::Cheermote::Type::Unknown;
        const auto& jType = jAction["type"];
        if (!jType.isNull() && jType.isString()) {
          if (jType == "global_first_party") {
            type = BitsConfiguration::Cheermote::Type::FirstParty;
          } else if (jType == "global_third_party") {
            type = BitsConfiguration::Cheermote::Type::ThirdParty;
          } else if (jType == "channel_custom") {
            type = BitsConfiguration::Cheermote::Type::Custom;
          }
        }
        action.type = type;

        cheermotes.push_back(std::move(action));
      }

      mResult = std::make_shared<Result>();
      mResult->cheermotes = std::move(cheermotes);
    }
  } else {
    ttv::trace::Message("ChatGetBitsConfigTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatGetBitsConfigTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
