/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatgetblocklisttask.h"

#include "twitchsdk/chat/internal/graphql/generated/getchatblocklistqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/utilities/userinfo.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/jsonobjectdescriptions.h"
#include "twitchsdk/core/json/reader.h"

#include <sstream>

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}  // namespace

ttv::chat::ChatGetBlockListTask::ChatGetBlockListTask(
  UserId userId, const std::string& authToken, std::shared_ptr<Result> result, Callback callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(callback), mResult(result), mUserId(userId) {
  TTV_ASSERT(userId != 0);
  TTV_ASSERT(result != nullptr);

  result->valid = false;

  ttv::trace::Message("ChatGetBlockListTask", MessageLevel::Info, "ChatGetBlockListTask created");
}

void ttv::chat::ChatGetBlockListTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.httpReqType = HTTP_POST_REQUEST;
  requestInfo.url = kHostName;
  ttv::json::Value root;
  root["query"] = chat::graphql::GetChatBlockListQueryInfo::kQuery;
  root["variables"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["id"] = std::to_string(mUserId);

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
}

void ttv::chat::ChatGetBlockListTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message("ChatGetBlockListTask", MessageLevel::Error,
        "Inside ChatGetBlockListTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    chat::graphql::GetChatBlockListQueryInfo::PayloadType payload;

    if (ttv::json::ToObject(jsonVal["data"], payload)) {
      for (const auto& user : payload.blockedUsers) {
        if (user.HasValue()) {
          mResult->users.emplace_back(graphql::GQLUserToUserInfo(user.Value()));
        } else {
          ttv::trace::Message(GetTaskName(), MessageLevel::Error, "Error parsing 'user' JSON");
        }
      }

      mResult->valid = true;
    } else {
      ttv::trace::Message("ChatGetBlockListTask", MessageLevel::Error,
        "Inside ChatGetBlockListTask::ProcessResponse - Invalid blocks element");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatGetBlockListTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatGetBlockListTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
