/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatgetchannelvodcommentsettingstask.h"

#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/types/coretypes.h"

namespace {
const char* kHostName = "https://api.twitch.tv/kraken/videos/channels/";
const char* kVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// curl -X GET -H 'Accept: application/vnd.twitchtv.v5+json' -H 'Client-ID: $CLIENTID' -H 'Authorization: OAuth $OAUTH'
// "https://api.twitch.tv/kraken/videos/channels/$CHANNELID/settings" | python -m json.tool GET
// /v5/videos/channels/:channel_id/settings

ttv::chat::ChatGetChannelVodCommentSettingsTask::ChatGetChannelVodCommentSettingsTask(
  ChannelId channelId, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(std::move(callback)), mChannelId(channelId) {
  TTV_ASSERT(mCallback);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetChannelVodCommentSettingsTask created");
}

void ttv::chat::ChatGetChannelVodCommentSettingsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << kHostName << mChannelId << "/settings/";

  Uri url(ss.str());

  requestInfo.requestHeaders.emplace_back("Accept", kVersion);
  requestInfo.url = url.GetUrl();
  requestInfo.httpReqType = HTTP_GET_REQUEST;
}

void ttv::chat::ChatGetChannelVodCommentSettingsTask::ProcessResponse(
  uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    ttv::json::Value jsonVal;
    ttv::json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message("ChatGetChannelVodCommentSettingsTask", MessageLevel::Error,
        "Inside ChatGetChannelVodCommentSettingsTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    if (!ParseChannelVodCommentSettingsJson(jsonVal, mResult)) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatGetChannelVodCommentSettingsTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatGetChannelVodCommentSettingsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }
    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
