/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatgetcommentrepliestask.h"

#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/types/coretypes.h"

namespace {
const char* kHostName = "https://api.twitch.tv/kraken/videos/comments/";
const char* kVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// curl -X GET -H 'Accept: application/vnd.twitchtv.v5+json' -H 'Client-ID: $CLIENTID'
// "https://api.twitch.tv/kraken/videos/comments/$COMMENTID/replies?limit=$LIMIT" | python -m json.tool GET
// /v5/videos/comments/:comment_id/replies

ttv::chat::ChatGetCommentRepliesTask::ChatGetCommentRepliesTask(const std::string& commentId,
  const TokenizationOptions& options, const std::shared_ptr<BitsConfiguration>& config, Callback&& callback)
    : HttpTask(nullptr, nullptr, nullptr),
      mBitsConfiguration(config),
      mTokenizationOptions(options),
      mCommentId(commentId),
      mCallback(std::move(callback)) {
  TTV_ASSERT(mCommentId.size() > 0);
  TTV_ASSERT(mCallback);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetCommentRepliesTask created");
}

void ttv::chat::ChatGetCommentRepliesTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << kHostName << mCommentId << "/replies/";

  Uri url(ss.str());

  // Comments currently only have a max of 50 replies, grab all of them.
  url.SetParam("limit", 50);

  requestInfo.requestHeaders.emplace_back("Accept", kVersion);
  requestInfo.url = url.GetUrl();
  requestInfo.httpReqType = HTTP_GET_REQUEST;
}

void ttv::chat::ChatGetCommentRepliesTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    ttv::json::Value jsonVal;
    ttv::json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatGetCommentRepliesTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    const auto& jComments = jsonVal["comments"];
    if (jComments.isNull() || !jComments.isArray()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatGetCommentRepliesTask::ProcessResponse - Invalid 'comments' element");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    for (const auto& jMessage : jComments) {
      mResult.emplace_back();

      if (!ParseChatCommentJson(jMessage, mTokenizationOptions, mBitsConfiguration, mLocalUserNames, mResult.back())) {
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        mResult.pop_back();
        return;
      }
    }
  } else {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatGetCommentRepliesTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }
    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
