/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatgetcommenttask.h"

#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/types/coretypes.h"

namespace {
const char* kHostName = "https://api.twitch.tv/kraken/videos/comments/";
const char* kVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// curl -X GET -H 'Accept: application/vnd.twitchtv.v5+json' -H 'Client-ID: $CLIENTID'
// "https://api.twitch.tv/kraken/videos/comments/$COMMENTID" | python -m json.tool GET /v5/videos/comments/:comment_id

ttv::chat::ChatGetCommentTask::ChatGetCommentTask(const std::string& commentId, const TokenizationOptions& options,
  const std::shared_ptr<BitsConfiguration>& config, Callback&& callback)
    : HttpTask(nullptr, nullptr, nullptr),
      mBitsConfiguration(config),
      mTokenizationOptions(options),
      mCommentId(commentId),
      mCallback(std::move(callback)) {
  TTV_ASSERT(mCommentId.size() > 0);
  TTV_ASSERT(mCallback);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetCommentTask created");
}

void ttv::chat::ChatGetCommentTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  Uri url;
  url.SetHostName(kHostName);
  url.SetPath(mCommentId);

  requestInfo.requestHeaders.emplace_back("Accept", kVersion);
  requestInfo.url = url.GetUrl();
  requestInfo.httpReqType = HTTP_GET_REQUEST;
}

void ttv::chat::ChatGetCommentTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    ttv::json::Value jsonVal;
    ttv::json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        "ChatGetCommentTask", MessageLevel::Error, "Inside ChatGetCommentTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    if (!ParseChatCommentJson(jsonVal, mTokenizationOptions, mBitsConfiguration, mLocalUserNames, mResult)) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatGetCommentTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatGetCommentTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }
    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
