/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatgetemoticonstask.h"

#include "twitchsdk/chat/internal/graphql/generated/fetchuseremoticonsetsqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

#include <sstream>

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// User emoticon sets:
// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "query($userId: ID!) { user(id: $userId) { emoteSets { id emotes {
// id
// token setID } } } }", "variables": { "userId": "$userId" }
//}' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatGetEmoticonsTask::ChatGetEmoticonsTask(UserId userId, const std::string& oauthToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, oauthToken.c_str()), mCallback(std::move(callback)), mUserId(userId) {
  TTV_ASSERT(mUserId != 0);
  TTV_ASSERT(oauthToken != "");

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetEmoticonsTask created for a user");
}

void ttv::chat::ChatGetEmoticonsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;
  root["query"] = graphql::GetFetchUserEmoticonSetsQuery();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["userId"] = std::to_string(mUserId);

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatGetEmoticonsTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatGetEmoticonsTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    if (!ParseUserEmoticonSets(jsonVal, mResult)) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatGetEmoticonsTask::ProcessResponse - Parsing json values failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatGetEmoticonsTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatGetEmoticonsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
