/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatgetvodcommentstask.h"

#include "twitchsdk/chat/chattypes.h"
#include "twitchsdk/chat/internal/graphql/generated/videocommentsrequestinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/internal/graphql/utilities/graphqlutilities.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/types/coretypes.h"

#include <sstream>

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
const uint32_t kDefaultChatResponseLimit = 60;
const uint32_t kCommentBatchDeltaTime = 250;
}  // namespace

ttv::chat::ChatGetVodCommentsTask::ChatGetVodCommentsTask(const std::string& vodId, const std::string& cursorUrl,
  const TokenizationOptions& options, const std::shared_ptr<BitsConfiguration>& config, Callback&& callback)
    : HttpTask(nullptr, nullptr, nullptr),
      mCursorUrl(cursorUrl),
      mVodId(vodId),
      mTimestampMilliseconds(0),
      mLimit(kDefaultChatResponseLimit),
      mBitsConfiguration(config),
      mTokenizationOptions(options),
      mCallback(std::move(callback)) {
  TTV_ASSERT(vodId.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetVodCommentsTask created");
}

ttv::chat::ChatGetVodCommentsTask::ChatGetVodCommentsTask(const std::string& vodId, uint64_t timestampMilliseconds,
  const TokenizationOptions& options, const std::shared_ptr<BitsConfiguration>& config, Callback&& callback)
    : HttpTask(nullptr, nullptr, nullptr),
      mVodId(vodId),
      mTimestampMilliseconds(timestampMilliseconds),
      mLimit(kDefaultChatResponseLimit),
      mBitsConfiguration(config),
      mTokenizationOptions(options),
      mCallback(std::move(callback)) {
  TTV_ASSERT(vodId.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatGetVodCommentsTask created");
}

void ttv::chat::ChatGetVodCommentsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.httpReqType = HTTP_POST_REQUEST;
  requestInfo.url = kHostName;
  ttv::json::Value root;

  root["query"] = ttv::core::graphql::VideoCommentsQueryInfo::kQuery;
  root["variables"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["videoID"] = mVodId;
  root["variables"]["limit"] = mLimit;
  root["variables"]["replyLimit"] = mLimit;
  root["variables"]["cursor"] = mCursorUrl;
  root["variables"]["contentOffsetSeconds"] = mTimestampMilliseconds / 1000;

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
}

void ttv::chat::ChatGetVodCommentsTask::ResponseCallback(uint statusCode, const std::vector<char>& response) {
  if (!mAborted) {
    if (statusCode >= 200 && statusCode < 300) {
      ProcessResponse(statusCode, response);
    } else if (statusCode == 401) {
      mTaskStatus = TTV_EC_AUTHENTICATION;
    } else if (statusCode == 404) {
      mTaskStatus = TTV_EC_NOT_AVAILABLE;
    } else {
      std::string msg(response.data(), response.size());
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "HTTP request failed with status code %d. Message: %s",
        statusCode, msg.c_str());
      mTaskStatus = TTV_EC_API_REQUEST_FAILED;
    }
  } else {
    mTaskStatus = TTV_EC_REQUEST_ABORTED;
  }
}

void ttv::chat::ChatGetVodCommentsTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  using namespace ttv::core::graphql;

  if (response.size() == 0) {
    ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return;
  }

  // Parse the returned JSON
  ttv::json::Value jsonVal;
  ttv::json::Reader jsonReader;
  bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
  if (!parseRet) {
    ttv::trace::Message(
      GetTaskName(), MessageLevel::Error, "Inside ChatGetVodCommentsTask::ProcessResponse - JSON parsing failed");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return;
  }

  if (jsonVal["errors"].isNonNullArray()) {
    mTaskStatus = TTV_EC_API_REQUEST_FAILED;
    const auto& errorsArray = jsonVal["errors"];
    for (const auto& error : errorsArray) {
      if (error["message"].isString()) {
        mResult.errorMsg += error["message"].asString();
        mResult.errorMsg += '\n';
      }
    }
    return;
  }

  VideoCommentsQueryInfo::PayloadType payload;

  if (!ttv::json::ToObject(jsonVal["data"], payload)) {
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return;
  }

  if (!(payload.comments.HasValue() && payload.comments.Value().edges.HasValue())) {
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    return;
  }

  std::vector<Result::CommentsBatch> commentBatches;
  Result::CommentsBatch currentBatch;
  uint64_t currentBatchTimestampMilliseconds = 0;
  bool firstMessage = true;

  // Parsing each comment and adding it to currentBatch and then commentBatches
  for (const auto& optionalComment : payload.comments.Value().edges.Value()) {
    if (optionalComment.HasValue() && optionalComment.Value().cursor.HasValue()) {
      mResult.nextCursorUrl = optionalComment.Value().cursor.Value();
    }

    if (!(optionalComment.HasValue() && optionalComment.Value().node.HasValue())) {
      continue;
    }

    const auto& comment = optionalComment.Value().node.Value();

    if (!(comment.video.HasValue() && comment.video.Value().owner.HasValue())) {
      continue;
    }

    ttv::ChannelId channelId = ttv::graphql::GQLUserIdToChannelId(comment.video.Value().owner.Value().id);

    ChatComment message;

    ParseChatCommentJsonGQL<VideoCommentsQueryInfo::VideoComment, VideoCommentsQueryInfo::VideoCommentState,
      VideoCommentsQueryInfo::VideoCommentSource>(
      comment, mTokenizationOptions, mBitsConfiguration, mLocalUserNames, message, channelId, mVodId, "");

    if (firstMessage) {
      firstMessage = false;
      currentBatchTimestampMilliseconds = message.timestampMilliseconds;
      currentBatch.comments.push_back(std::move(message));
      continue;
    }

    if (message.timestampMilliseconds >= currentBatchTimestampMilliseconds &&
        message.timestampMilliseconds < currentBatchTimestampMilliseconds + kCommentBatchDeltaTime) {
      currentBatch.comments.push_back(std::move(message));
    } else if (message.timestampMilliseconds > currentBatchTimestampMilliseconds + kCommentBatchDeltaTime) {
      if (currentBatch.comments.size() > 0) {
        currentBatch.baseTimestampMilliseconds = currentBatchTimestampMilliseconds;
        commentBatches.push_back(currentBatch);
        currentBatch = Result::CommentsBatch();
      }
      currentBatch.comments.push_back(std::move(message));
      currentBatchTimestampMilliseconds = message.timestampMilliseconds;
    }
  }  // for (const auto& optionalComment : payload.comments.Value().edges.Value())

  if (currentBatch.comments.size() > 0)  // put last list in batch, and complete buffer
  {
    currentBatch.baseTimestampMilliseconds = currentBatchTimestampMilliseconds;
    commentBatches.push_back(std::move(currentBatch));
  }

  mResult.commentsListBatches = std::move(commentBatches);
}

void ttv::chat::ChatGetVodCommentsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }
    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
