/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatjson.h"

#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"
#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/internal/graphql/utilities/graphqlutilities.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/systemclock.h"

#include <inttypes.h>
#include <stdlib.h>
#include <string.h>

#include <sstream>
#include <utility>

namespace {
const char* kActionPrefix = "/me ";

struct Range {
  uint32_t start;
  uint32_t end;
};

// Need this on Windows to avoid truncating the decorated name (as it is longer than the compiler limit).
// Reference: https://msdn.microsoft.com/en-us/library/074af4b6.aspx
using RangeMap = std::map<std::string, std::vector<Range>>;
using PairList = std::vector<std::pair<std::string, std::string>>;

/**
 * The expected output for an empty RangeMap is an empty string
 * Given the following single input:
  * ((anonymous namespace)::RangeMap) $24 = size=2 {
      [0] = {
        first = "27509"
        second = size=1 {
          [0] = (start = 13, end = 21)
        }
      }
    }
 * The expected output is: "27509:13-21"
 * Given the following multiple input:
 * ((anonymous namespace)::RangeMap) $24 = size=2 {
      [0] = {
        first = "27509"
        second = size=1 {
          [0] = (start = 13, end = 21)
        }
      }
      [1] = {
        first = "81249"
        second = size=2 {
          [0] = (start = 0, end = 11)
          [1] = (start = 23, end = 34)
        }
      }
    }
 * The expected output is: "27509:13-21/81249:0-11,23-34"
 */
std::string CreateEmoteTagsFromRangeMap(const RangeMap& rangeMap) {
  std::stringstream sstream;
  bool first = true;
  for (auto kvp : rangeMap) {
    if (!first) {
      sstream << "/";
    }
    first = false;

    sstream << kvp.first << ":";

    const std::vector<Range>& ranges = kvp.second;
    for (size_t i = 0; i < ranges.size(); ++i) {
      if (i > 0) {
        sstream << ",";
      }
      sstream << ranges[i].start << "-" << ranges[i].end;
    }
  }
  return sstream.str();
}
}  // namespace

bool ttv::chat::ParseRaidStatusJson(const ttv::json::Value& jRaid, RaidStatus& status) {
  const auto& jRaidId = jRaid["id"];
  const auto& jCreatorChannelId = jRaid["creator_id"];
  const auto& jSourceChannelId = jRaid["source_id"];
  const auto& jTargetChannelId = jRaid["target_id"];
  const auto& jTargetUserLogin = jRaid["target_login"];
  const auto& jTargetUserDisplayName = jRaid["target_display_name"];
  const auto& jTargetUserProfileImageUrl = jRaid["target_profile_image"];
  const auto& jViewerCount = jRaid["viewer_count"];
  const auto& jTransitionJitterSeconds = jRaid["transition_jitter_seconds"];
  const auto& jForceRaidNowSeconds = jRaid["force_raid_now_seconds"];

  if (!jRaidId.isString() || !jCreatorChannelId.isString() || !jSourceChannelId.isString() ||
      !jTargetChannelId.isString() || !jTargetUserLogin.isString() || !jTargetUserDisplayName.isString() ||
      !jTargetUserProfileImageUrl.isString() || !jViewerCount.isNumeric() || !jTransitionJitterSeconds.isNumeric() ||
      !jForceRaidNowSeconds.isNumeric()) {
    return false;
  }

  status.raidId = jRaidId.asString();
  status.targetUserLogin = jTargetUserLogin.asString();
  status.targetUserDisplayName = jTargetUserDisplayName.asString();
  status.targetUserProfileImageUrl = jTargetUserProfileImageUrl.asString();

  int numRead = ParseChannelId(jCreatorChannelId, status.creatorUserId);
  if (numRead != 1 || status.creatorUserId == 0) {
    return false;
  }

  numRead = ParseChannelId(jSourceChannelId, status.sourceChannelId);
  if (numRead != 1 || status.sourceChannelId == 0) {
    return false;
  }

  numRead = ParseChannelId(jTargetChannelId, status.targetChannelId);
  if (numRead != 1 || status.targetChannelId == 0) {
    return false;
  }

  status.numUsersInRaid = static_cast<uint32_t>(jViewerCount.asUInt());
  status.transitionJitterSeconds = static_cast<uint32_t>(jTransitionJitterSeconds.asUInt());
  status.forceRaidNowSeconds = static_cast<uint32_t>(jForceRaidNowSeconds.asUInt());

  return true;
}

bool ttv::chat::ParseMessageJson(const ttv::json::Value& jMessage, const TokenizationOptions& tokenizationOptions,
  const std::shared_ptr<BitsConfiguration>& bitsConfig, const std::vector<std::string>& localUserNames,
  WhisperMessage& result) {
  bool parsed = true;

  const auto& jMessageId = jMessage["id"];
  result.messageId = static_cast<MessageId>(jMessageId.asUInt());

  const auto& jThreadId = jMessage["thread_id"];
  if (!jThreadId.isNull() && jThreadId.isString()) {
    result.threadId = jThreadId.asString();
  }

  const auto& jFromId = jMessage["from_id"];
  result.messageInfo.userId = static_cast<UserId>(jFromId.asUInt());

  const auto& jSent = jMessage["sent_ts"];
  ParseTimestamp(jSent, result.messageInfo.timestamp);

  ParseString(jMessage, "message_id", result.messageUuid);

  result.messageInfo.nameColorARGB = 0xFF000000;

  std::string message;

  const auto& jBody = jMessage["body"];
  if (jBody.isString()) {
    message = jBody.asString();

    static std::string actionPrefix(kActionPrefix);

    // We need to manually handle actions that are indicated as starting with "/me "
    if (StartsWith(message, actionPrefix)) {
      message = message.substr(actionPrefix.size());
      result.messageInfo.flags.action = true;
    }
  }

  const auto& jTags = jMessage["tags"];
  if (!jTags.isNull() && jTags.isObject()) {
    const auto& jLogin = jTags["login"];
    if (jLogin.isString()) {
      result.messageInfo.userName = jLogin.asString();
    }

    const auto& jDisplayName = jTags["display_name"];
    if (jDisplayName.isString() && jDisplayName.asString().length() > 0) {
      result.messageInfo.displayName = jDisplayName.asString();
    } else if (jLogin.isString()) {
      // This is a hack to handle cases where the backend sends an empty display_name
      result.messageInfo.displayName = jLogin.asString();
    }

    bool useDefaultColor = true;
    const auto& jColor = jTags["color"];
    if (jColor.isString()) {
      useDefaultColor = !ParseColor(jColor.asCString(), result.messageInfo.nameColorARGB);
    }

    if (useDefaultColor) {
      result.messageInfo.nameColorARGB = GetRandomUserColor(result.messageInfo.userName);
    }

    const auto& jUserType = jTags["user_type"];
    if (jUserType.isString()) {
      result.messageInfo.userMode = ParseUserType(jUserType.asString());
    }

    const auto& jBadges = jTags["badges"];
    if (jBadges.isArray() && jBadges.size() > 0) {
      PairList pairs;

      for (const auto& jBadge : jBadges) {
        const auto& jBadgeSetId = jBadge["id"];
        const auto& jBadgeSetVersion = jBadge["version"];

        if (jBadgeSetId.isString() && jBadgeSetVersion.isString()) {
          pairs.emplace_back(jBadgeSetId.asString(), jBadgeSetVersion.asString());
        }
      }

      std::transform(pairs.begin(), pairs.end(), std::back_inserter(result.messageInfo.badges),
        [](const std::pair<std::string, std::string>& pair) {
          MessageBadge badge;
          badge.name = pair.first;
          badge.version = pair.second;
          return badge;
        });
    }

    std::string emotesMessageTag;

    if (tokenizationOptions.emoticons) {
      const auto& jEmotes = jTags["emotes"];

      // Recreate the emotes message tag in the format
      //   1:2-3,5-6/2:9-15
      if (jEmotes.isArray() && jEmotes.size() > 0) {
        RangeMap rangeMap;
        for (const auto& jEmote : jEmotes) {
          std::string emoteId;
          const auto& jStart = jEmote["start"];
          const auto& jEnd = jEmote["end"];

          if (ParseEmoticonId(jEmote, "id", emoteId) && jStart.isNumeric() && jEnd.isNumeric()) {
            std::vector<Range>& rangeSet = rangeMap[emoteId];

            Range range;
            range.start = static_cast<uint32_t>(jStart.asUInt());
            range.end = static_cast<uint32_t>(jEnd.asUInt());
            rangeSet.emplace_back(range);
          }
        }

        emotesMessageTag = CreateEmoteTagsFromRangeMap(rangeMap);
      }
    }

    // NOTE: This assumes that you can't send/receive Bits over private messages
    TokenizationOptions editedTokenizationOptions = tokenizationOptions;
    if (bitsConfig == nullptr) {
      editedTokenizationOptions.bits = false;
    }
    ttv::chat::TokenizeServerMessage(
      message, editedTokenizationOptions, emotesMessageTag, bitsConfig, localUserNames, result.messageInfo);
  }

  return parsed;
}

bool ttv::chat::ParseChatCommentJson(const ttv::json::Value& jMessage, const TokenizationOptions& tokenizationOptions,
  const std::shared_ptr<BitsConfiguration>& bitsConfig, const std::vector<std::string>& localUserNames,
  ChatComment& result) {
  if (!ParseString(jMessage, "_id", result.commentId)) {
    return false;
  }

  const auto& jChannelId = jMessage["channel_id"];
  ParseChannelId(jChannelId, result.channelId);

  ParseString(jMessage, "parent_id", result.parentCommentId);
  ParseString(jMessage, "content_id", result.contentId);

  if (jMessage["content_offset_seconds"].isDouble()) {
    double contentOffsetSeconds;
    ParseDouble(jMessage, "content_offset_seconds", contentOffsetSeconds);
    if (contentOffsetSeconds < 0) {
      contentOffsetSeconds = 0;
    }
    result.timestampMilliseconds = static_cast<uint64_t>(contentOffsetSeconds * 1000);
  } else {
    int64_t contentOffsetSeconds;
    ParseInt(jMessage, "content_offset_seconds", contentOffsetSeconds);
    if (contentOffsetSeconds < 0) {
      contentOffsetSeconds = 0;
    }
    result.timestampMilliseconds = static_cast<uint64_t>(contentOffsetSeconds) * 1000;
  }

  ParseTimestamp(jMessage, "created_at", result.messageInfo.timestamp);
  ParseTimestamp(jMessage, "updated_at", result.updatedAt);

  const auto& jPublishedState = jMessage["state"];
  if (jPublishedState.isString()) {
    std::string publishedStateString = jPublishedState.asString();
    if (publishedStateString == "published") {
      result.publishedState = ChatCommentPublishedState::Published;
    } else if (publishedStateString == "unpublished") {
      result.publishedState = ChatCommentPublishedState::Unpublished;
    } else if (publishedStateString == "pending_review") {
      result.publishedState = ChatCommentPublishedState::PendingReview;
    } else if (publishedStateString == "pending_review_spam") {
      result.publishedState = ChatCommentPublishedState::PendingReviewSpam;
    } else {
      result.publishedState = ChatCommentPublishedState::Unknown;
    }
  }

  const auto& jSource = jMessage["source"];
  if (jSource.isString()) {
    std::string sourceString = jSource.asString();
    if (sourceString == "comment") {
      result.commentSource = ChatCommentSource::Comment;
    } else if (sourceString == "chat") {
      result.commentSource = ChatCommentSource::Chat;
    } else {
      result.commentSource = ChatCommentSource::Unknown;
    }
  }

  ParseBool(jMessage, "more_replies", result.moreReplies, false);

  const auto& jCommenter = jMessage["commenter"];
  if (!jCommenter.isNull() && jCommenter.isObject()) {
    const auto& jUserId = jCommenter["_id"];
    if (jUserId.isString()) {
      ParseUserId(jUserId, result.messageInfo.userId);
    }

    ParseString(jCommenter, "name", result.messageInfo.userName);
    ParseString(jCommenter, "display_name", result.messageInfo.displayName);
  }

  const auto& jMessageInfo = jMessage["message"];
  if (!jMessageInfo.isNull() && jMessageInfo.isObject()) {
    bool useDefaultColor = true;
    const auto& jColor = jMessageInfo["user_color"];
    if (jColor.isString()) {
      useDefaultColor = !ParseColor(jColor.asCString(), result.messageInfo.nameColorARGB);
    }

    if (useDefaultColor) {
      result.messageInfo.nameColorARGB = GetRandomUserColor(result.messageInfo.userName);
    }

    const auto& jBadges = jMessageInfo["user_badges"];
    if (!jBadges.isNull() && jBadges.isArray()) {
      for (const auto& jBadge : jBadges) {
        MessageBadge badge;

        if (ParseString(jBadge, "_id", badge.name) && ParseString(jBadge, "version", badge.version)) {
          result.messageInfo.badges.emplace_back(badge);
        }
      }
    }

    std::string emotesMessageTag;

    if (tokenizationOptions.emoticons) {
      const auto& jEmotes = jMessageInfo["emoticons"];

      // Recreate the emotes message tag in the format
      //   1:2-3,5-6/2:9-15
      if (!jEmotes.isNull() && jEmotes.isArray()) {
        RangeMap rangeMap;
        // Iterate over the emoticons and build the RangeMap
        for (const auto& jEmote : jEmotes) {
          std::string parsedEmoteId;
          Range range;
          if (ParseEmoticonId(jEmote, "_id", parsedEmoteId) && ParseUInt(jEmote, "begin", range.start) &&
              ParseUInt(jEmote, "end", range.end)) {
            rangeMap[parsedEmoteId].emplace_back(range);
          }
        }

        emotesMessageTag = CreateEmoteTagsFromRangeMap(rangeMap);
      }
    }

    std::string messageBody;
    ParseString(jMessageInfo, "body", messageBody);
    TokenizeServerMessage(
      messageBody, tokenizationOptions, emotesMessageTag, bitsConfig, localUserNames, result.messageInfo);
  }

  return true;
}

// allow only supported templates to be defined or else compile time errors
template void ttv::chat::ParseChatCommentJsonGQL<ttv::core::graphql::VideoCommentsQueryInfo::VideoComment,
  ttv::core::graphql::VideoCommentsQueryInfo::VideoCommentState,
  ttv::core::graphql::VideoCommentsQueryInfo::VideoCommentSource>(
  const ttv::core::graphql::VideoCommentsQueryInfo::VideoComment&, const TokenizationOptions&,
  const std::shared_ptr<BitsConfiguration>&, const std::vector<std::string>&, ChatComment&, ttv::ChannelId,
  const std::string&, const std::string&);
template void ttv::chat::ParseChatCommentJsonGQL<ttv::core::graphql::CreateVideoCommentMutationQueryInfo::VideoComment,
  ttv::core::graphql::CreateVideoCommentMutationQueryInfo::VideoCommentState,
  ttv::core::graphql::CreateVideoCommentMutationQueryInfo::VideoCommentSource>(
  const ttv::core::graphql::CreateVideoCommentMutationQueryInfo::VideoComment&, const TokenizationOptions&,
  const std::shared_ptr<BitsConfiguration>&, const std::vector<std::string>&, ChatComment&, ttv::ChannelId,
  const std::string&, const std::string&);

template <typename VideoComment, typename VideoCommentState, typename VideoCommentSource>
void ttv::chat::ParseChatCommentJsonGQL(const VideoComment& comment, const TokenizationOptions& tokenizationOptions,
  const std::shared_ptr<BitsConfiguration>& bitsConfig, const std::vector<std::string>& localUserNames,
  ChatComment& message, ttv::ChannelId channelId, const std::string& videoId, const std::string& parentCommentId) {
  if (!comment.commenter.HasValue()) {
    return;
  }

  const auto& commenter = comment.commenter.Value();

  message.commentId = comment.id;
  message.channelId = channelId;

  if (channelId == 0 && comment.video.HasValue() && comment.video.Value().owner.HasValue()) {
    message.channelId = ttv::graphql::GQLUserIdToChannelId(comment.video.Value().owner.Value().id);
  }

  message.contentId = videoId;
  if (videoId.empty() && comment.video.HasValue()) {
    message.contentId = comment.video.Value().id;
  }

  message.timestampMilliseconds = uint64_t(comment.contentOffsetSeconds) * 1000;

  message.messageInfo.timestamp = comment.createdAt;
  message.updatedAt = comment.updatedAt;

  message.parentCommentId = parentCommentId;

  switch (comment.state) {
    case VideoCommentState::DELETED:
      message.publishedState = ChatCommentPublishedState::Deleted;
      break;
    case VideoCommentState::PENDING_REVIEW:
      message.publishedState = ChatCommentPublishedState::PendingReview;
      break;
    case VideoCommentState::PENDING_REVIEW_SPAM:
      message.publishedState = ChatCommentPublishedState::PendingReviewSpam;
      break;
    case VideoCommentState::PUBLISHED:
      message.publishedState = ChatCommentPublishedState::Published;
      break;
    case VideoCommentState::UNPUBLISHED:
      message.publishedState = ChatCommentPublishedState::Unpublished;
      break;

    default:
      TTV_ASSERT(false, "ParseChatJsonGQL", "Unhandled VideoCommentState encountered");
      message.publishedState = ChatCommentPublishedState::Unknown;
      break;
  }

  switch (comment.source) {
    case VideoCommentSource::CHAT:
      message.commentSource = ChatCommentSource::Chat;
      break;
    case VideoCommentSource::COMMENT:
      message.commentSource = ChatCommentSource::Comment;
      break;
    case VideoCommentSource::UNKNOWN:
      message.commentSource = ChatCommentSource::Unknown;
      break;

    default:
      TTV_ASSERT(false, "ParseChatJsonGQL", "Unhandled VideoCommentSource encountered");
      message.commentSource = ChatCommentSource::Unknown;
      break;
  }

  message.messageInfo.userName = commenter.login.ValueOrDefault("");
  message.messageInfo.displayName = commenter.displayName.ValueOrDefault("");
  message.messageInfo.userId = ttv::graphql::GQLUserIdToChannelId(commenter.id);

  if (comment.message.HasValue()) {
    bool useDefaultColor = true;
    const auto& color = comment.message.Value().userColor.ValueOrDefault("");
    useDefaultColor = !ParseColor(color, message.messageInfo.nameColorARGB);

    if (useDefaultColor) {
      message.messageInfo.nameColorARGB = ttv::chat::GetRandomUserColor(message.messageInfo.userName);
    }

    if (comment.message.Value().userBadges.HasValue()) {
      for (const auto& optionalUserBadge : comment.message.Value().userBadges.Value()) {
        if (!optionalUserBadge.HasValue()) {
          continue;
        }

        const auto& userBadge = optionalUserBadge.Value();

        MessageBadge badge;
        badge.name = userBadge.setID;
        badge.version = userBadge.version;

        message.messageInfo.badges.push_back(std::move(badge));
      }
    }

    std::string messageBody;
    RangeMap rangeMap;
    if (comment.message.Value().fragments.HasValue()) {
      for (const auto& optionalFragment : comment.message.Value().fragments.Value()) {
        if (!optionalFragment.HasValue()) {
          continue;
        }

        const auto& fragment = optionalFragment.Value();
        messageBody += fragment.text;

        // Recreate the emotes message tag in the format
        //   1:2-3,5-6/2:9-15
        if (fragment.emote.HasValue()) {
          std::string parsedEmoteId;
          Range range;

          parsedEmoteId = fragment.emote.Value().emoteID.ValueOrDefault("");
          range.start = static_cast<uint32_t>(fragment.emote.Value().from.ValueOrDefault(0));
          range.end = static_cast<uint32_t>(fragment.emote.Value().to.ValueOrDefault(0));
          rangeMap[parsedEmoteId].push_back(std::move(range));
        }
      }

      std::string emotesMessageTag = CreateEmoteTagsFromRangeMap(rangeMap);
      ttv::chat::TokenizeServerMessage(
        messageBody, tokenizationOptions, emotesMessageTag, bitsConfig, localUserNames, message.messageInfo);
    }
  }
}

bool ttv::chat::ParseChannelVodCommentSettingsJson(
  const ttv::json::Value& jMessage, ChannelVodCommentSettings& result) {
  const auto& jChannelId = jMessage["channel_id"];
  if (!ParseChannelId(jChannelId, result.channelId)) {
    return false;
  }

  ParseTimestamp(jMessage, "created_at", result.createdAt);
  ParseTimestamp(jMessage, "updated_at", result.updatedAt);

  const auto& jPublishingMode = jMessage["publishing_mode"];
  if (jPublishingMode.isString()) {
    std::string publishingModeString = jPublishingMode.asString();
    if (publishingModeString == "open" || publishingModeString == "") {
      result.publishingMode = CommentPublishingMode::Open;
    } else if (publishingModeString == "review") {
      result.publishingMode = CommentPublishingMode::Review;
    } else if (publishingModeString == "disabled") {
      result.publishingMode = CommentPublishingMode::Disabled;
    } else {
      result.publishingMode = CommentPublishingMode::Unknown;
      return false;
    }
  }

  if (!ParseUInt(jMessage, "followers_only_duration_seconds", result.followersOnlyDurationSeconds)) {
    return false;
  }

  return true;
}

bool ttv::chat::ParseGraphQLErrorCode(
  const ttv::json::Value& jVal, const std::string& mutationName, ttv::chat::GraphQLErrorCode& code) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jMutationPayload = jData[mutationName];
  if (jMutationPayload.isNull() || !jMutationPayload.isObject()) {
    return false;
  }

  const auto& jError = jMutationPayload["error"];
  if (!jError.isNull() && jError.isObject()) {
    const auto& jCode = jError["code"];
    if (jCode.isNull() || !jCode.isString()) {
      return false;
    }

    ttv::json::ToObject(jCode, code);
    return false;
  }

  return true;
}

bool ttv::chat::ParseEditRoomMessage(const ttv::json::Value& jVal, ChatRoomMessage& result) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jMutationPayload = jData["editRoomMessage"];
  if (jMutationPayload.isNull() || !jMutationPayload.isObject()) {
    return false;
  }

  const auto& jMessage = jMutationPayload["message"];
  if (jMessage.isNull() || !jMessage.isObject()) {
    return false;
  }

  if (!ttv::json::ObjectSchema<json::description::GraphQLChatRoomMessage>::Parse(jMessage, result)) {
    return false;
  }

  return true;
}

bool ttv::chat::ParseSendRoomMessage(
  const ttv::json::Value& jVal, ChatRoomMessage& result, SendRoomMessageError& error) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jMutationPayload = jData["sendRoomMessage"];
  if (jMutationPayload.isNull() || !jMutationPayload.isObject()) {
    return false;
  }

  const auto& jMessage = jMutationPayload["message"];
  if (jMessage.isNull() || !jMessage.isObject()) {
    // Check for extra error information
    const auto& jGraphQLError = jMutationPayload["error"];
    if (!jGraphQLError.isNull() && jGraphQLError.isObject()) {
      ttv::json::ToObject(jGraphQLError, error);
    }

    return false;
  }

  if (!ttv::json::ObjectSchema<json::description::GraphQLChatRoomMessage>::Parse(jMessage, result)) {
    return false;
  }

  return true;
}

bool ttv::chat::ParseRoomMessages(const ttv::json::Value& jVal, std::vector<ChatRoomMessage>& resultMessages,
  std::string& resultCursor, bool& resultMoreMessages) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jRoom = jData["room"];
  if (jRoom.isNull() || !jRoom.isObject()) {
    return false;
  }

  const auto& jMessages = jRoom["messages"];
  if (jMessages.isNull() || !jMessages.isObject()) {
    return false;
  }

  const auto& jEdges = jMessages["edges"];
  if (jEdges.isNull() || !jEdges.isArray()) {
    return false;
  }

  for (const auto& jEdge : jEdges) {
    ParseString(jEdge, "cursor", resultCursor);

    const auto& jMessage = jEdge["node"];
    if (jMessage.isNull() || !jMessage.isObject()) {
      return false;
    }

    ChatRoomMessage message;

    if (ttv::json::ObjectSchema<json::description::GraphQLChatRoomMessage>::Parse(jMessage, message)) {
      resultMessages.emplace_back(std::move(message));
    }
  }

  const auto& jPageInfo = jMessages["pageInfo"];
  if (jPageInfo.isNull() || !jPageInfo.isObject()) {
    return false;
  }

  ParseBool(jPageInfo, "hasNextPage", resultMoreMessages, true);

  return true;
}

bool ttv::chat::ParseCreateRoom(const ttv::json::Value& jVal, ChatRoomInfo& result, CreateRoomError& error) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jPayload = jData["createRoom"];
  if (jPayload.isNull() || !jPayload.isObject()) {
    return false;
  }

  const auto& jRoom = jPayload["room"];
  if (jRoom.isNull() || !jRoom.isObject()) {
    // Check for extra error information
    const auto& jGraphQLError = jPayload["error"];
    if (!jGraphQLError.isNull() && jGraphQLError.isObject()) {
      ttv::json::ToObject(jGraphQLError, error);
    }

    return false;
  }

  return ttv::json::ObjectSchema<json::description::GraphQLChatRoomInfo>::Parse(jRoom, result);
}

bool ttv::chat::ParseUpdateRoom(const ttv::json::Value& jVal, ChatRoomInfo& result, UpdateRoomError& error) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jPayload = jData["updateRoom"];
  if (jPayload.isNull() || !jPayload.isObject()) {
    return false;
  }

  const auto& jRoom = jPayload["room"];
  if (jRoom.isNull() || !jRoom.isObject()) {
    // Check for extra error information
    const auto& jGraphQLError = jPayload["error"];
    if (!jGraphQLError.isNull() && jGraphQLError.isObject()) {
      ttv::json::ToObject(jGraphQLError, error);
    }

    return false;
  }

  return ttv::json::ObjectSchema<json::description::GraphQLChatRoomInfo>::Parse(jRoom, result);
}

bool ttv::chat::ParseUpdateRoomModes(const ttv::json::Value& jVal, ChatRoomInfo& result, UpdateRoomModesError& error) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jPayload = jData["updateRoomModes"];
  if (jPayload.isNull() || !jPayload.isObject()) {
    return false;
  }

  const auto& jRoom = jPayload["room"];
  if (jRoom.isNull() || !jRoom.isObject()) {
    // Check for extra error information
    const auto& jGraphQLError = jPayload["error"];
    if (!jGraphQLError.isNull() && jGraphQLError.isObject()) {
      ttv::json::ToObject(jGraphQLError, error);
    }

    return false;
  }

  return ttv::json::ObjectSchema<json::description::GraphQLChatRoomInfo>::Parse(jRoom, result);
}

bool ttv::chat::ParseRoomInfo(const ttv::json::Value& jVal, ChatRoomInfo& result) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jRoom = jData["room"];
  if (jRoom.isNull() || !jRoom.isObject()) {
    return false;
  }

  return ttv::json::ObjectSchema<json::description::GraphQLChatRoomInfo>::Parse(jRoom, result);
}

bool ttv::chat::ParseGraphQLChatRooms(const ttv::json::Value& jVal, std::vector<ChatRoomInfo>& result) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jUser = jData["user"];
  if (jUser.isNull() || !jUser.isObject()) {
    return false;
  }

  const auto& jRooms = jUser["channelRooms"];
  if (jRooms.isNull() || !jRooms.isArray()) {
    return false;
  }

  for (const auto& jRoom : jRooms) {
    if (jRoom.isNull() || !jRoom.isObject()) {
      return false;
    }

    ChatRoomInfo info;
    if (!ttv::json::ObjectSchema<json::description::GraphQLChatRoomInfo>::Parse(jRoom, info)) {
      return false;
    }

    result.emplace_back(std::move(info));
  }

  return true;
}

bool ttv::chat::ParseGraphQLChatRoomView(
  const ttv::json::Value& jVal, const std::string& mutationName, ChatRoomInfo& result) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jMutationPayload = jData[mutationName];
  if (jMutationPayload.isNull() || !jMutationPayload.isObject()) {
    return false;
  }

  const auto& jView = jMutationPayload["roomView"];
  if (jView.isNull() || !jView.isObject()) {
    return false;
  }

  const auto& jRoom = jView["room"];
  if (jRoom.isNull() || !jRoom.isObject()) {
    return false;
  }

  if (!ttv::json::ObjectSchema<json::description::GraphQLChatRoomInfo>::Parse(jRoom, result)) {
    return false;
  }

  return true;
}

bool ttv::chat::ParseGraphQLUserMods(
  const ttv::json::Value& jVal, std::vector<std::string>& result, std::string& nextCursor) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jUser = jData["user"];
  if (jUser.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jMods = jUser["mods"];
  if (jMods.isNull() || !jMods.isObject()) {
    return false;
  }

  const auto& jEdges = jMods["edges"];
  if (jEdges.isNull() || !jEdges.isArray()) {
    return false;
  }

  for (const auto& jMod : jEdges) {
    ParseString(jMod, "cursor", nextCursor);

    const auto& jNode = jMod["node"];
    if (jMod.isNull() || !jMod.isObject()) {
      continue;
    }

    std::string modName;
    if (ParseString(jNode, "login", modName)) {
      result.push_back(modName);
    }
  }

  const auto& jPageInfo = jMods["pageInfo"];
  if (jPageInfo.isNull() || !jPageInfo.isObject()) {
    return false;
  }

  bool moreModerators;
  ParseBool(jPageInfo, "hasNextPage", moreModerators, false);
  if (!moreModerators) {
    nextCursor = "";
  }

  return true;
}

bool ttv::chat::ParseBanUser(const ttv::json::Value& jVal, BanUserError& error) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jMutationPayload = jData["banUserFromChatRoom"];
  if (jMutationPayload.isNull() || !jMutationPayload.isObject()) {
    return false;
  }

  const auto& jBan = jMutationPayload["ban"];
  if (jBan.isNull() || !jBan.isObject()) {
    // Check for extra error information
    const auto& jGraphQLError = jMutationPayload["error"];
    if (!jGraphQLError.isNull() && jGraphQLError.isObject()) {
      ttv::json::ToObject(jGraphQLError, error);
    }

    return false;
  }

  return true;
}

bool ttv::chat::ParseUnbanUser(const ttv::json::Value& jVal, UnbanUserError& error) {
  const auto& jErrors = jVal["errors"];
  if (!jErrors.isNull()) {
    return false;
  }

  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jMutationPayload = jData["unbanUserFromChatRoom"];
  if (jMutationPayload.isNull() || !jMutationPayload.isObject()) {
    return false;
  }

  const auto& jBan = jMutationPayload["ban"];
  if (jBan.isNull() || !jBan.isObject()) {
    // Check for extra error information
    const auto& jGraphQLError = jMutationPayload["error"];
    if (!jGraphQLError.isNull() && jGraphQLError.isObject()) {
      ttv::json::ToObject(jGraphQLError, error);
    }

    return false;
  }

  return true;
}

bool ttv::chat::ParseUserEmoticonSets(const ttv::json::Value& jVal, std::vector<EmoticonSet>& result) {
  const auto& jData = jVal["data"];
  if (jData.isNull() || !jData.isObject()) {
    return false;
  }

  const auto& jUser = jData["user"];
  if (jUser.isNull() || !jUser.isObject()) {
    return false;
  }

  const auto& jEmoteSets = jUser["emoteSets"];
  if (jEmoteSets.isNull() || !jEmoteSets.isArray()) {
    return false;
  }

  for (const auto& jEmoteSet : jEmoteSets) {
    EmoticonSet set;

    const auto& owner = jEmoteSet["owner"];
    if (owner.isNonNullObject()) {
      ParseString(owner, "displayName", set.ownerDisplayName);
    }

    if (!ParseEmoticonId(jEmoteSet, "id", set.emoticonSetId)) {
      break;
    }

    const auto& jEmotes = jEmoteSet["emotes"];
    if (jEmotes.isNull() || !jEmotes.isArray()) {
      break;
    }

    for (const auto& jEmote : jEmotes) {
      Emoticon emote;
      if (ttv::json::ToObject(jEmote, emote)) {
        set.emoticons.emplace_back(emote);
      }
    }

    if (!set.emoticons.empty()) {
      result.emplace_back(set);
    }
  }

  return (!result.empty());
}
