/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatmodusertask.h"

#include "twitchsdk/chat/internal/graphql/generated/moduserqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/generated/unmoduserqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($input: ModUserInput!) {modUser(input:$input) { target {
// id } }", "variables": { "input" : { "targetLogin": "$LOGINNAME", "channelID": "$CHANNELID" }}
// }' "https://gql.twitch.tv/gql" | python -m json.tool

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($input: ModUserInput!) {modUser(input:$input) { target {
// id } }", "variables": { "input" : { "targetLogin": "$LOGINNAME", "channelID": "$CHANNELID" }}
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatModUserTask::ChatModUserTask(
  UserId userId, const std::string& modUserName, const std::string& authToken, ModCallback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mModCallback(std::move(callback)),
      mModUserName(modUserName),
      mUserId(userId),
      mGrantMod(true) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(modUserName.size() > 0);
  TTV_ASSERT(userId != 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatModUserTask created for modding");
}

ttv::chat::ChatModUserTask::ChatModUserTask(
  UserId userId, const std::string& modUserName, const std::string& authToken, UnmodCallback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mUnmodCallback(std::move(callback)),
      mModUserName(modUserName),
      mUserId(userId),
      mGrantMod(false) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(modUserName.size() > 0);
  TTV_ASSERT(userId != 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatModUserTask created for unmodding");
}

void ttv::chat::ChatModUserTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  if (mGrantMod) {
    root["query"] = graphql::GetModUserMutation();
  } else {
    root["query"] = graphql::GetUnmodUserMutation();
  }
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["input"] = json::Value(json::objectValue);
  root["variables"]["input"]["channelID"] = std::to_string(mUserId);
  root["variables"]["input"]["targetLogin"] = mModUserName;

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatModUserTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatModUserTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    GraphQLErrorCode code = GraphQLErrorCode::SUCCESS;
    std::string mutationName;

    if (mGrantMod) {
      mutationName = "modUser";
    } else {
      mutationName = "unmodUser";
    }

    if (!ParseGraphQLErrorCode(jsonVal, mutationName, code)) {
      if (code == GraphQLErrorCode::SUCCESS) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatRoomSendMessageTask::ProcessResponse - Parsing json values failed");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      } else {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatRoomSendMessageTask::ProcessResponse - GraphQL Error received");
        mTaskStatus = TTV_EC_GRAPHQL_ERROR;
      }
    }

    if (mGrantMod) {
      mModError.code = code;
    } else {
      mUnmodError.code = code;
    }
  } else {
    ttv::trace::Message("ChatModUserTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatModUserTask::OnComplete() {
  if (mAborted) {
    mTaskStatus = TTV_EC_REQUEST_ABORTED;
  }

  if (mGrantMod && mModCallback) {
    mModCallback(this, mTaskStatus.ec, std::move(mModError));
  } else if (!mGrantMod && mUnmodCallback) {
    mUnmodCallback(this, mTaskStatus.ec, std::move(mUnmodError));
  }
}
