/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatpostcommentreplytask.h"

#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/types/coretypes.h"

namespace {
const char* kHostName = "https://api.twitch.tv/kraken/videos/comments/";
const char* kVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// curl -X POST -H 'Accept: application/vnd.twitchtv.v5+json' -H 'Client-ID: $CLIENTID' -H 'Authorization: OAuth $OAUTH'
// -d 'message=$MESSAGE' "https://api.twitch.tv/kraken/videos/comments/$COMMENTID/replies" | python -m json.tool POST
// /v5/videos/comments/:comment_id/replies

ttv::chat::ChatPostCommentReplyTask::ChatPostCommentReplyTask(const std::string& commentId, const std::string& message,
  const TokenizationOptions& options, const std::shared_ptr<BitsConfiguration>& config, const std::string& authToken,
  Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mBitsConfiguration(config),
      mTokenizationOptions(options),
      mCommentId(commentId),
      mMessage(message),
      mCallback(std::move(callback)) {
  TTV_ASSERT(mCommentId.size() > 0);
  TTV_ASSERT(mMessage.size() > 0);
  TTV_ASSERT(mCallback);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatPostCommentReplyTask created");
}

void ttv::chat::ChatPostCommentReplyTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << kHostName << mCommentId << "/replies";

  Uri url(ss.str());

  ttv::json::Value root;

  root["message"] = mMessage;

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.requestHeaders.emplace_back("Accept", kVersion);
  requestInfo.requestHeaders.emplace_back("Content-Type", "application/json");
  requestInfo.url = url.GetUrl();
  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatPostCommentReplyTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    ttv::json::Value jsonVal;
    ttv::json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message("ChatPostCommentReplyTask", MessageLevel::Error,
        "Inside ChatPostCommentReplyTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    if (!ParseChatCommentJson(jsonVal, mTokenizationOptions, mBitsConfiguration, mLocalUserNames, mResult)) {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatPostCommentReplyTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatPostCommentReplyTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }
    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
