/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatpostcommenttask.h"

#include "twitchsdk/chat/internal/graphql/generated/postchatcommentrequestinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/types/coretypes.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}  // namespace

ttv::chat::ChatPostCommentTask::ChatPostCommentTask(const std::string& vodId, const std::string& message,
  uint64_t timestampMilliseconds, const TokenizationOptions& options, const std::shared_ptr<BitsConfiguration>& config,
  const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mBitsConfiguration(config),
      mTokenizationOptions(options),
      mVodId(vodId),
      mMessage(message),
      mCallback(std::move(callback)),
      mTimestampMilliseconds(timestampMilliseconds) {
  TTV_ASSERT(mMessage.size() > 0);
  TTV_ASSERT(mVodId.size() > 0);
  TTV_ASSERT(mCallback);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatPostCommentTask created");
}

void ttv::chat::ChatPostCommentTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.httpReqType = HTTP_POST_REQUEST;
  requestInfo.url = kHostName;
  json::Value root;

  root["query"] = core::graphql::CreateVideoCommentMutationQueryInfo::kQuery;
  root["variables"] = json::Value(ttv::json::objectValue);
  root["variables"]["message"] = mMessage;
  root["variables"]["contentOffsetSeconds"] = mTimestampMilliseconds / 1000;
  root["variables"]["videoID"] = mVodId;

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
}

void ttv::chat::ChatPostCommentTask::ResponseCallback(uint statusCode, const std::vector<char>& response) {
  using namespace ttv::core::graphql;

  if (!mAborted) {
    if (statusCode == 401) {
      mTaskStatus = TTV_EC_AUTHENTICATION;
    } else {
      if (response.size() > 0) {
        ttv::json::Value jsonVal;
        ttv::json::Reader jsonReader;
        bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
        if (!parseRet) {
          ttv::trace::Message("ChatPostCommentTask", MessageLevel::Error,
            "Inside ChatPostCommentTask::ProcessResponse - JSON parsing failed");
          mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
          return;
        }

        if (statusCode >= 200 && statusCode < 300) {
          const auto& jErrors = jsonVal["errors"];
          if (!jErrors.isNull()) {
            ttv::trace::Message(GetTaskName(), MessageLevel::Error,
              "Inside ChatPostCommentTask::ProcessResponse - query returned with error");
            mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
            return;
          }

          CreateVideoCommentMutationQueryInfo::PayloadType payload;
          if (!ttv::json::ToObject(jsonVal["data"], payload)) {
            mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
            return;
          }

          const auto& gqlComment = payload.comment;
          ParseChatCommentJsonGQL<CreateVideoCommentMutationQueryInfo::VideoComment,
            CreateVideoCommentMutationQueryInfo::VideoCommentState,
            CreateVideoCommentMutationQueryInfo::VideoCommentSource>(
            gqlComment, mTokenizationOptions, mBitsConfiguration, mLocalUserNames, mResult);
        } else {
          std::string msg(response.data(), response.size());
          ttv::trace::Message(GetTaskName(), MessageLevel::Error,
            "HTTP request failed with status code %d. Message: %s", statusCode, msg.c_str());

          mTaskStatus = TTV_EC_API_REQUEST_FAILED;
          const auto& errorJson = jsonVal["error"];
          if (!errorJson.isNull()) {
            ttv::ParseString(errorJson, "message", mErrorMessage);
          }
        }
      } else {
        ttv::trace::Message("ChatPostCommentTask", MessageLevel::Error, "No response body");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      }
    }
  } else {
    mTaskStatus = TTV_EC_REQUEST_ABORTED;
  }
}

void ttv::chat::ChatPostCommentTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }
    mCallback(this, mTaskStatus.ec, std::move(mResult), std::move(mErrorMessage));
  }
}
