/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatpropertiestask.h"

#include "twitchsdk/chat/internal/graphql/generated/getchannelpropertiesqueryinfo.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/socket.h"

#include <sstream>

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}  // namespace

ttv::chat::ChatPropertiesTask::Result::Result() : chat_delay_duration(0), hide_chat_links(false) {}

ttv::chat::ChatPropertiesTask::ChatPropertiesTask(ChannelId channelId, Callback callback)
    : HttpTask(nullptr, nullptr, nullptr), mChannelId(channelId), mCallback(callback) {
  ttv::trace::Message("ChatPropertiesTask", MessageLevel::Info, "ChatPropertiesTask created");
}

void ttv::chat::ChatPropertiesTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.httpReqType = HTTP_POST_REQUEST;
  requestInfo.url = kHostName;
  ttv::json::Value root;
  root["query"] = chat::graphql::GetChannelPropertiesQueryInfo::kQuery;
  root["variables"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["id"] = std::to_string(mChannelId);

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
}

void ttv::chat::ChatPropertiesTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message(
        "ChatPropertiesTask", MessageLevel::Error, "Inside ChatPropertiesTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }

    chat::graphql::GetChannelPropertiesQueryInfo::PayloadType payload;

    if (ttv::json::ToObject(jsonVal["data"], payload)) {
      mResult = std::make_shared<Result>();

      if (payload.chatSettings.HasValue()) {
        const auto& chatSettings = payload.chatSettings.Value();

        mResult->chat_rules = chatSettings.rules;
        mResult->chat_delay_duration = static_cast<uint32_t>(chatSettings.chatDelayMs / 1000);
        mResult->hide_chat_links = chatSettings.blockLinks;
      } else {
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      }
    } else {
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatPropertiesTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatPropertiesTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, mResult);
  }
}
