/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatraidtask.h"

#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"

#include <sstream>

// Join a raid
//   curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/raids/$RAIDID/join" | python -m json.tool POST
//   /v5/raids/:raid_id/join
//
// Leave a raid
//   curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/raids/$RAIDID/leave" | python -m json.tool POST
//   /v5/raids/:raid_id/leave
//
// Create a raid
//   curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H "Accept:
//   application/vnd.twitchtv.v5+json"
//   "https://api.twitch.tv/kraken/raids?source_id=$SOURCEUSERID&target_id=$TARGETUSERID" | python -m json.tool POST
//   /v5/raids
//
// Cancel a raid
//   curl -X DELETE  -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H "Accept:
//   application/vnd.twitchtv.v5+json" "https://api.twitch.tv/kraken/raids?source_id=$SOURCEUSERID" | python -m
//   json.tool DELETE /v5/raids

namespace {
const char* kApiVersion = "application/vnd.twitchtv.v5+json";
}

ttv::chat::ChatRaidTask::ChatRaidTask(UserId sourceChannelId, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mSourceChannelId(sourceChannelId),
      mTargetChannelId(0),
      mChangeType(ChangeType::Invalid) {
  TTV_ASSERT(sourceChannelId != 0);

  ttv::trace::Message("ChatRaidTask", MessageLevel::Info, "ChatRaidTask created");
}

void ttv::chat::ChatRaidTask::Join(const std::string& raidId) {
  mChangeType = ChangeType::Join;
  mRaidId = raidId;
}

void ttv::chat::ChatRaidTask::Leave(const std::string& raidId) {
  mChangeType = ChangeType::Leave;
  mRaidId = raidId;
}

void ttv::chat::ChatRaidTask::Start(UserId targetChannelId) {
  TTV_ASSERT(targetChannelId != 0);
  mChangeType = ChangeType::Start;
  mTargetChannelId = targetChannelId;
}

void ttv::chat::ChatRaidTask::RaidNow() {
  mChangeType = ChangeType::RaidNow;
}

void ttv::chat::ChatRaidTask::Cancel(UserId targetChannelId) {
  TTV_ASSERT(targetChannelId != 0);
  mChangeType = ChangeType::Cancel;
  mTargetChannelId = targetChannelId;
}

void ttv::chat::ChatRaidTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << "https://api.twitch.tv/kraken/raids/";

  switch (mChangeType) {
    case ChangeType::Join:
      ss << UrlEncode(mRaidId) << "/join";
      requestInfo.httpReqType = HTTP_POST_REQUEST;
      break;
    case ChangeType::Leave:
      ss << UrlEncode(mRaidId) << "/leave";
      requestInfo.httpReqType = HTTP_POST_REQUEST;
      break;
    case ChangeType::Start:
      requestInfo.httpReqType = HTTP_POST_REQUEST;
      break;
    case ChangeType::RaidNow:
      ss << "go/";
      requestInfo.httpReqType = HTTP_POST_REQUEST;
      break;
    case ChangeType::Cancel:
      requestInfo.httpReqType = HTTP_DELETE_REQUEST;
      break;
    default:
      TTV_ASSERT(mChangeType != ChangeType::Invalid);
      break;
  }

  Uri url(ss.str());

  if (mChangeType == ChangeType::Start || mChangeType == ChangeType::Cancel || mChangeType == ChangeType::RaidNow) {
    url.SetParam("source_id", mSourceChannelId);

    if (mChangeType == ChangeType::Start) {
      url.SetParam("target_id", mTargetChannelId);
    }
  }

  requestInfo.url = url;
  requestInfo.requestHeaders.emplace_back(HttpParam("Accept", kApiVersion));
}

void ttv::chat::ChatRaidTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (mChangeType == ChangeType::Join || mChangeType == ChangeType::Leave || mChangeType == ChangeType::Cancel ||
      mChangeType == ChangeType::RaidNow) {
    mTaskStatus = TTV_EC_SUCCESS;
  } else if (mChangeType == ChangeType::Start) {
    if (response.size() > 0) {
      ttv::json::Value root;

      bool parsed = ParseDocument(response, root);
      if (!parsed) {
        ttv::trace::Message(
          GetTaskName(), MessageLevel::Error, "Inside ChatRaidTask::ProcessResponse - JSON parsing failed");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
        return;
      }

      if (!parsed || root.isNull() || !root.isObject()) {
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      } else {
        mTaskStatus = TTV_EC_SUCCESS;
      }
    } else {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error, "No response body");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    TTV_ASSERT(false);
  }
}

void ttv::chat::ChatRaidTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec);
  }
}
