/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatroomdeletemessagetask.h"

#include "twitchsdk/chat/internal/graphql/generated/deleteroommessagequeryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($message: DeleteRoomMessageInput!)
// {deleteRoomMessage(input:$message) { message { id } } }", "variables": { "message" : { "roomID": "$ROOMID",
// "messageID": "$MESSAGEID" }}
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatRoomDeleteMessageTask::ChatRoomDeleteMessageTask(
  const std::string& roomId, const std::string& messageId, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mRoomId(roomId),
      mMessageId(messageId) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(roomId.size() > 0);
  TTV_ASSERT(messageId.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatRoomDeleteMessageTask created");
}

void ttv::chat::ChatRoomDeleteMessageTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = graphql::GetDeleteRoomMessageMutation();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["message"] = json::Value(json::objectValue);
  root["variables"]["message"]["roomID"] = mRoomId;
  root["variables"]["message"]["messageID"] = mMessageId;

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatRoomDeleteMessageTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatRoomDeleteMessageTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    const auto& jErrors = jsonVal["errors"];
    if (!jErrors.isNull()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatRoomDeleteMessageTask::ProcessResponse - query returned with error");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatRoomDeleteMessageTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatRoomDeleteMessageTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec);
  }
}
