/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatroomdeleteroomtask.h"

#include "twitchsdk/chat/internal/graphql/generated/deleteroomqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($input: DeleteRoomInput!) {deleteRoom(input:$input) { room
// { id } } }", "variables": { "input" : { "roomID": "$ROOMID" }}
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatRoomDeleteRoomTask::ChatRoomDeleteRoomTask(
  const std::string& roomId, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(std::move(callback)), mRoomId(roomId) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(roomId.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatRoomDeleteRoomTask created");
}

void ttv::chat::ChatRoomDeleteRoomTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = graphql::GetDeleteRoomMutation();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["input"] = json::Value(json::objectValue);
  root["variables"]["input"]["roomID"] = mRoomId;

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatRoomDeleteRoomTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatRoomDeleteRoomTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    const auto& jErrors = jsonVal["errors"];
    if (!jErrors.isNull()) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatRoomDeleteRoomTask::ProcessResponse - query returned with error");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatRoomDeleteRoomTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatRoomDeleteRoomTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec);
  }
}
