/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatroomfetchmessagestask.h"

#include "twitchsdk/chat/internal/graphql/generated/fetchroommessagesaftercursorqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/generated/fetchroommessagesaftertimestampqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/generated/fetchroommessagesbeforecursorqueryinfo.h"
#include "twitchsdk/chat/internal/graphql/generated/fetchroommessagesbeforetimestampqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// Backward pagination, fetches older messages than cursor.
// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "query($roomId: ID!, $limit: Int, $cursor: Cursor) {room(id:$roomId)
// { messages (last: $limit, before: $cursor){ edges { cursor node { id content { fragments { text content { ... on
// Emote { id } __typename } } } } } pageInfo { hasNextPage hasPreviousPage } } } }", "variables": { "roomId":
// "$ROOMID", "cursor":"$CURSOR", "limit": "$LIMIT" }
// }' "https://gql.twitch.tv/gql" | python -m json.tool

// Forward pagination, fetches newer messages than the timestamp.
// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "query($roomId: ID!, $limit: Int, $timestamp: Time)
// {room(id:$roomId) { messages (first: $limit, fromTime: $timestamp){ edges { cursor node { id content { fragments {
// text content { ... on Emote { id } __typename } } } } } pageInfo { hasNextPage hasPreviousPage } } } }", "variables":
// { "roomId": "$ROOMID", "timestamp":"$TIMESTAMP", "limit": "$LIMIT" }
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatRoomFetchMessagesTask::ChatRoomFetchMessagesTask(const std::string& roomId, ChannelId channelId,
  bool fetchAfter, uint32_t limit, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mRoomId(roomId),
      mChannelId(channelId),
      mLimit(limit),
      mResultMoreMessages(false),
      mFetchAfter(fetchAfter),
      mUseTimestamp(false) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(mRoomId.size() > 0);
  TTV_ASSERT(channelId != 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatRoomFetchMessagesTask created");
}

void ttv::chat::ChatRoomFetchMessagesTask::SetCursor(const std::string& cursor) {
  mUseTimestamp = false;
  mVariableValue = cursor;
}

void ttv::chat::ChatRoomFetchMessagesTask::SetTime(Timestamp timestamp) {
  mUseTimestamp = true;
  mVariableValue = UnixTimestampToRFC3339String(timestamp);
}

void ttv::chat::ChatRoomFetchMessagesTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  json::Value root;
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["roomId"] = mRoomId;
  root["variables"]["limit"] = mLimit;
  root["variables"]["channelId"] = std::to_string(mChannelId);

  if (mUseTimestamp) {
    root["variables"]["timestamp"] = mVariableValue;
    if (mFetchAfter) {
      root["query"] = graphql::GetFetchRoomMessagesAfterTimestampQuery();
    } else {
      root["query"] = graphql::GetFetchRoomMessagesBeforeTimestampQuery();
    }
  } else {
    root["variables"]["cursor"] = mVariableValue;
    if (mFetchAfter) {
      root["query"] = graphql::GetFetchRoomMessagesAfterCursorQuery();
    } else {
      root["query"] = graphql::GetFetchRoomMessagesBeforeCursorQuery();
    }
  }

  requestInfo.url = kHostName;

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatRoomFetchMessagesTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatRoomFetchMessagesTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    if (!ParseRoomMessages(jsonVal, mResultMessages, mResultCursor, mResultMoreMessages)) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatRoomFetchMessagesTask::ProcessResponse - Parsing json values failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatRoomFetchMessagesTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatRoomFetchMessagesTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mResultMessages), std::move(mResultCursor), mResultMoreMessages);
  }
}
