/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatroomupdateinfotask.h"

#include "twitchsdk/chat/internal/graphql/generated/updateroomqueryinfo.h"
#include "twitchsdk/chat/internal/json/chatjsonobjectdescriptions.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($info: UpdateRoomInput!) {updateRoom(input:$info) { room {
// id name topic rolePermissions { read send } self { lastReadAt isMuted isArchived isUnread unreadMentionCount }
//  owner { id login displayName description profileImageURL(width: 300) createdAt } } } }",
// "variables": { "info" : { "roomID": "$ROOMID", "name": "$NAME", "topic": "$TOPIC", "minimumReadMessagesRole":
// "$MINIMUMREADROLE", "minimumSendMessagesRole": "$MINIUMSENDROLE" } }
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatRoomUpdateInfoTask::ChatRoomUpdateInfoTask(
  const std::string& roomId, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(std::move(callback)), mRoomId(roomId) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(roomId.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatRoomUpdateInfoTask created");
}

void ttv::chat::ChatRoomUpdateInfoTask::SetName(const std::string& name) {
  mRoomName = name;
  mInput = "name";
}

void ttv::chat::ChatRoomUpdateInfoTask::SetTopic(const std::string& topic) {
  mRoomTopic = topic;
  mInput = "topic";
}

void ttv::chat::ChatRoomUpdateInfoTask::SetRoomRolePermissions(RoomRolePermissions permissions) {
  mPermissions = permissions;
  mInput = "permissions";
}

void ttv::chat::ChatRoomUpdateInfoTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  ttv::json::Value root;

  root["query"] = graphql::GetUpdateRoomMutation();
  root["variables"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["input"] = ttv::json::Value(ttv::json::objectValue);
  root["variables"]["input"]["roomID"] = mRoomId;

  if (mInput == "name") {
    root["variables"]["input"][mInput] = mRoomName;
  } else if (mInput == "topic") {
    root["variables"]["input"][mInput] = mRoomTopic;
  } else if (mInput == "permissions") {
    ToJson(mPermissions.read, root["variables"]["input"]["minimumReadMessagesRole"]);
    ToJson(mPermissions.send, root["variables"]["input"]["minimumSendMessagesRole"]);
  }

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatRoomUpdateInfoTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    ttv::json::Value jsonVal;
    ttv::json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatRoomUpdateInfoTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    if (!ParseUpdateRoom(jsonVal, mResult, mError)) {
      if (mError.code == GraphQLErrorCode::SUCCESS) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatRoomUpdateInfoTask::ProcessResponse - Parsing json values failed");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      } else {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatRoomUpdateInfoTask::ProcessResponse - GraphQL Error received");
        mTaskStatus = TTV_EC_GRAPHQL_ERROR;
      }
    }
  } else {
    ttv::trace::Message("ChatRoomUpdateInfoTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatRoomUpdateInfoTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mError), std::move(mResult));
  }
}
