/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatroomupdatemodetask.h"

#include "twitchsdk/chat/internal/graphql/generated/updateroommodesqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($info: UpdateRoomModesInput!)
// {updateRoomModes(input:$info) { room { id name topic rolePermissions { read send } modes { slowModeDurationSeconds
// r9kModeEnabled emotesOnlyModeEnabled } self { lastReadAt isMuted isArchived isUnread unreadMentionCount } owner { id
// login displayName description profileImageURL(width: 300) createdAt } } } }", "variables": { "info" : { "roomID":
// "$ROOMID", "name": "$NAME", "enableSlowMode": "$SLOWMODE", "enableEmotesOnlyMode": "$EMOTESMODE", "enableR9KMode":
// "$R9KMODE" } }
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatRoomUpdateModeTask::ChatRoomUpdateModeTask(
  const std::string& roomId, ChatMode mode, bool turnOn, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mRoomId(roomId),
      mMode(mode),
      mSlowModeDurationSeconds(0),
      mTurnOn(turnOn) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(roomId.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatRoomUpdateModeTask created");
}

void ttv::chat::ChatRoomUpdateModeTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = graphql::GetUpdateRoomModesMutation();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["input"] = json::Value(json::objectValue);
  root["variables"]["input"]["roomID"] = mRoomId;

  switch (mMode) {
    case ChatMode::Slow: {
      root["variables"]["input"]["enableSlowMode"] = mTurnOn;
      if (mTurnOn) {
        root["variables"]["input"]["slowModeDurationSeconds"] = mSlowModeDurationSeconds;
      }
      break;
    }
    case ChatMode::R9k: {
      root["variables"]["input"]["enableR9KMode"] = mTurnOn;
      break;
    }
    case ChatMode::EmotesOnly: {
      root["variables"]["input"]["enableEmotesOnlyMode"] = mTurnOn;
      break;
    }
  }

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatRoomUpdateModeTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatRoomUpdateModeTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    if (!ParseUpdateRoomModes(jsonVal, mResult, mError)) {
      if (mError.code == GraphQLErrorCode::SUCCESS) {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatRoomUpdateModeTask::ProcessResponse - Parsing json values failed");
        mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      } else {
        ttv::trace::Message(GetTaskName(), MessageLevel::Error,
          "Inside ChatRoomUpdateModeTask::ProcessResponse - GraphQL Error received");
        mTaskStatus = TTV_EC_GRAPHQL_ERROR;
      }
    }
  } else {
    ttv::trace::Message("ChatRoomUpdateModeTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatRoomUpdateModeTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mError), std::move(mResult));
  }
}
