/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatroomupdateviewtask.h"

#include "twitchsdk/chat/internal/graphql/generated/updateroomviewqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($view: UpdateRoomViewInput!) {updateRoomView(input:$view)
// {roomView { lastReadAt isMuted isArchived isUnread unreadMentionCount } } }", "variables": { "view" : { "roomID":
// "$ROOMID" "lastReadAt": "$LASTREADAT", "isMuted": "$ISMUTED", "isArchived": "$ISARCHIVED" } }
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatRoomUpdateViewTask::ChatRoomUpdateViewTask(
  const std::string& roomId, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(std::move(callback)), mRoomId(roomId) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(roomId.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatRoomUpdateViewTask created");
}

void ttv::chat::ChatRoomUpdateViewTask::SetLastReadAt(Timestamp lastReadAt) {
  mVariableKey = "lastReadAt";
  mVariableValue = UnixTimestampToRFC3339String(lastReadAt);
}

void ttv::chat::ChatRoomUpdateViewTask::SetIsMuted(bool isMuted) {
  mVariableKey = "isMuted";
  mVariableValue = isMuted;
}

void ttv::chat::ChatRoomUpdateViewTask::SetIsArchived(bool isArchived) {
  mVariableKey = "isArchived";
  mVariableValue = isArchived;
}

void ttv::chat::ChatRoomUpdateViewTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = graphql::GetUpdateRoomViewMutation();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["input"] = json::Value(json::objectValue);
  root["variables"]["input"]["roomID"] = mRoomId;

  if (!mVariableKey.empty()) {
    root["variables"]["input"][mVariableKey] = mVariableValue;
  }

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatRoomUpdateViewTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatRoomUpdateViewTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    if (!ParseGraphQLChatRoomView(jsonVal, "updateRoomView", mResult)) {
      ttv::trace::Message(GetTaskName(), MessageLevel::Error,
        "Inside ChatRoomUpdateViewTask::ProcessResponse - Parsing json values failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }
  } else {
    ttv::trace::Message("ChatRoomUpdateViewTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatRoomUpdateViewTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec, std::move(mResult));
  }
}
