/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatsetchannelvodcommentsettingstask.h"

#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/types/coretypes.h"

namespace {
const char* kHostName = "https://api.twitch.tv/kraken/videos/channels/";
const char* kVersion = "application/vnd.twitchtv.v5+json";
}  // namespace

// curl -X PUT -H 'Accept: application/vnd.twitchtv.v5+json' -H 'Client-ID: $CLIENTID' -H 'Authorization: OAuth $OAUTH'
// -d 'publishing_mode=$MODE&followers_only_duration_seconds=$DURATION'
// "https://api.twitch.tv/kraken/videos/channels/$CHANNELID/settings" | python -m json.tool PUT
// /v5/videos/channels/:channel_id/settings

ttv::chat::ChatSetChannelVodCommentSettingsTask::ChatSetChannelVodCommentSettingsTask(ChannelId channelId,
  CommentPublishingMode mode, uint32_t followersOnlyDurationSeconds, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mChannelId(channelId),
      mFollowersOnlyDurationSeconds(followersOnlyDurationSeconds),
      mPublishingMode(mode),
      mSetFollowersDuration(true),
      mSetPublishingMode(true) {
  TTV_ASSERT(mCallback);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatSetChannelVodCommentSettingsTask created");
}

ttv::chat::ChatSetChannelVodCommentSettingsTask::ChatSetChannelVodCommentSettingsTask(
  ChannelId channelId, CommentPublishingMode mode, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mChannelId(channelId),
      mPublishingMode(mode),
      mSetFollowersDuration(false),
      mSetPublishingMode(true) {
  TTV_ASSERT(mCallback);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatSetChannelVodCommentSettingsTask created");
}

ttv::chat::ChatSetChannelVodCommentSettingsTask::ChatSetChannelVodCommentSettingsTask(
  ChannelId channelId, uint32_t followersOnlyDurationSeconds, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()),
      mCallback(std::move(callback)),
      mChannelId(channelId),
      mFollowersOnlyDurationSeconds(followersOnlyDurationSeconds),
      mSetFollowersDuration(true),
      mSetPublishingMode(false) {
  TTV_ASSERT(mCallback);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatSetChannelVodCommentSettingsTask created");
}

void ttv::chat::ChatSetChannelVodCommentSettingsTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  std::stringstream ss;
  ss << kHostName << mChannelId << "/settings/";

  Uri url(ss.str());

  ttv::json::Value root;

  if (mSetPublishingMode) {
    switch (mPublishingMode) {
      case CommentPublishingMode::Open: {
        root["publishing_mode"] = "open";
        break;
      }
      case CommentPublishingMode::Review: {
        root["publishing_mode"] = "review";
        break;
      }
      case CommentPublishingMode::Disabled: {
        root["publishing_mode"] = "disabled";
        break;
      }
      default: { break; }
    }
  }
  if (mSetFollowersDuration) {
    root["followers_only_duration_seconds"] = mFollowersOnlyDurationSeconds;
  }

  ttv::json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);
  requestInfo.requestHeaders.emplace_back("Accept", kVersion);
  requestInfo.requestHeaders.emplace_back("Content-Type", "application/json");
  requestInfo.url = url.GetUrl();
  requestInfo.httpReqType = HTTP_PUT_REQUEST;
}

void ttv::chat::ChatSetChannelVodCommentSettingsTask::ProcessResponse(
  uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    ttv::json::Value jsonVal;
    ttv::json::Reader jsonReader;
    bool parseRet = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parseRet) {
      ttv::trace::Message("ChatSetChannelVodCommentSettingsTask", MessageLevel::Error,
        "Inside ChatSetChannelVodCommentSettingsTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
      return;
    }
  } else {
    ttv::trace::Message("ChatSetChannelVodCommentSettingsTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatSetChannelVodCommentSettingsTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }
    mCallback(this, mTaskStatus.ec);
  }
}
