/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/task/chatupdatecolortask.h"

#include "twitchsdk/chat/internal/graphql/generated/updateuserchatcolorqueryinfo.h"
#include "twitchsdk/chat/internal/task/chatjson.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/json/writer.h"
#include "twitchsdk/core/stringutilities.h"

namespace {
const char* kHostName = "https://gql.twitch.tv/gql";
}

// curl -X POST -H "Client-Id: $CLIENTID" -H "Authorization: OAuth $OAUTH" -H 'Accept: application/json' -H
// 'Content-Type: application/json'  -d '{ "query": "mutation($input: UpdateChatColorInput!)
// {updateChatColor(input:$input) { user { id chatColor } }", "variables": { "input" : { "color": "$COLOR" }}
// }' "https://gql.twitch.tv/gql" | python -m json.tool

ttv::chat::ChatUpdateColorTask::ChatUpdateColorTask(
  const std::string& color, const std::string& authToken, Callback&& callback)
    : HttpTask(nullptr, nullptr, authToken.c_str()), mCallback(std::move(callback)), mColor(color) {
  TTV_ASSERT(authToken.size() > 0);
  TTV_ASSERT(color.size() > 0);

  ttv::trace::Message(GetTaskName(), MessageLevel::Info, "ChatUpdateColorTask created");
}

void ttv::chat::ChatUpdateColorTask::FillHttpRequestInfo(HttpRequestInfo& requestInfo) {
  requestInfo.url = kHostName;

  json::Value root;

  root["query"] = graphql::GetUpdateUserChatColorMutation();
  root["variables"] = json::Value(json::objectValue);
  root["variables"]["input"] = json::Value(json::objectValue);
  root["variables"]["input"]["color"] = mColor;

  json::FastWriter writer;
  requestInfo.requestBody = writer.write(root);

  requestInfo.httpReqType = HTTP_POST_REQUEST;
}

void ttv::chat::ChatUpdateColorTask::ProcessResponse(uint /*statusCode*/, const std::vector<char>& response) {
  if (response.size() > 0) {
    // Parse the returned JSON
    json::Value jsonVal;
    json::Reader jsonReader;
    bool parsed = jsonReader.parse(response.data(), response.data() + response.size(), jsonVal);
    if (!parsed) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatUpdateColorTask::ProcessResponse - JSON parsing failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }

    const auto& jErrors = jsonVal["errors"];
    if (!jErrors.isNull()) {
      ttv::trace::Message(
        GetTaskName(), MessageLevel::Error, "Inside ChatUpdateColorTask::ProcessResponse - Parsing json values failed");
      mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
    }
  } else {
    ttv::trace::Message("ChatUpdateColorTask", MessageLevel::Error, "No response body");
    mTaskStatus = TTV_EC_WEBAPI_RESULT_INVALID_JSON;
  }
}

void ttv::chat::ChatUpdateColorTask::OnComplete() {
  if (mCallback) {
    if (mAborted) {
      mTaskStatus = TTV_EC_REQUEST_ABORTED;
    }

    mCallback(this, mTaskStatus.ec);
  }
}
