/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/tokenranges.h"

#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/core/utf8.h"

namespace ttv {
namespace chat {
namespace tokenranges {
namespace {
RangeBase ToRangeBase(const AutoModFlagsRange& range) {
  return RangeBase(static_cast<int>(range.startIndex), static_cast<int>(range.endIndex));
}

RangeBase ToRangeBase(const TokenRange& range) {
  return RangeBase(range.startIndex, range.endIndex);
}

void MergeFromRangeBase(const RangeBase& range, AutoModFlagsRange& result) {
  result.startIndex = static_cast<uint32_t>(range.startIndex);
  result.endIndex = static_cast<uint32_t>(range.stopIndex);
}

void MergeFromRangeBase(const RangeBase& range, TokenRange& result) {
  result.startIndex = range.startIndex;
  result.endIndex = range.stopIndex;
}

// Returns a collection of RangeBases using the input range to extract the start and stop index
template <typename T>
std::vector<RangeBase> ToRangeBases(const std::vector<T>& ranges) {
  std::vector<RangeBase> rangeBase;
  for (const auto& range : ranges) {
    rangeBase.push_back(ToRangeBase(range));
  }
  return rangeBase;
}

// Returns an collection of ranges using the data from |originalRanges| and with modified start and stop indicies from
// |newRanges|
template <typename T>
std::vector<T> FromRangeBases(const std::vector<T>& originalRanges, const std::vector<RangeBase>& newRanges) {
  TTV_ASSERT(originalRanges.size() == newRanges.size());

  auto modifiedRanges{originalRanges};

  for (auto itr = std::make_pair(modifiedRanges.begin(), newRanges.begin());
       itr.first != modifiedRanges.end() && itr.second != newRanges.end(); itr.first++, itr.second++) {
    MergeFromRangeBase(*itr.second, *itr.first);
  }

  return modifiedRanges;
}

}  // namespace

std::vector<RangeBase> GetUtf8ToByteRanges(const std::vector<RangeBase>& utf8Ranges, const std::string& message) {
  // This assumes that the ranges are sorted by start index
  //
  // for utf8 we want the byte range to match the the first byte of the utf8 range
  // and the last byte of the utf8 range
  //
  // [b00, b01, b10, b20, b21 b23 b30 b31] bytes
  // [u0        u1   u2           u3     ] utf8
  // [          | byte range  |          ]
  // range [u1-u2] -> [b10-b23]
  //
  // algorithm
  // 1. Move to start utf8 count and get corresponding byte count
  // 2. Move to stop utf8 count and get corresponding byte count,
  //    add the length of current utf8 to get ending byte range
  int byteOffset = 0;
  int utf8Offset = 0;
  int sequenceLength = -1;
  const char* cur = message.data();

  auto ranges = utf8Ranges;

  for (auto& range : ranges) {
    while (utf8Offset < range.startIndex && sequenceLength != 0) {
      cur = AdvanceToNextUtf8Character(cur, sequenceLength);
      utf8Offset++;
      byteOffset += sequenceLength;
    }

    range.startIndex = byteOffset;

    while (utf8Offset < range.stopIndex && sequenceLength != 0) {
      cur = AdvanceToNextUtf8Character(cur, sequenceLength);
      utf8Offset++;
      byteOffset += sequenceLength;
    }

    AdvanceToNextUtf8Character(cur, sequenceLength);
    range.stopIndex = byteOffset + ((sequenceLength < 1) ? 0 : (sequenceLength - 1));
  }

  return ranges;
}

std::vector<RangeBase> GetByteToUtf8Ranges(const std::vector<RangeBase>& utf8Ranges, const std::string& message) {
  // This assumes that the ranges are sorted by start index
  //
  // from |bytes| we want the |utf8| range to match the first byte to the utf8 index, the last byte should match to the
  // utf8 index in that range
  //
  // [b00 b01 |b10, b20, b21 b23| b30 b31] bytes
  // [u0       u1   u2           u3      ] utf8
  // [         |    |                    ] new range
  // range [b10-b23] -> [u1-u2]
  //
  // algorithm
  // 1. Move to start utf8 count and get corresponding byte count
  // 2. Move to stop utf8 count and get corresponding byte count,
  //    add the length of current utf8 to get ending byte range
  int byteOffset = 0;
  int utf8Offset = 0;
  int sequenceLength = -1;
  const char* cur = message.data();

  auto ranges = utf8Ranges;

  for (auto& range : ranges) {
    while (byteOffset < range.startIndex && sequenceLength != 0) {
      cur = AdvanceToNextUtf8Character(cur, sequenceLength);
      utf8Offset++;
      byteOffset += sequenceLength;
    }

    // we need to account for multi byte utf8, ideally |startIndex| should always point to the first value in the byte
    // range so we should always have range.startIndex == byteOffset.  Handling unexpected case
    range.startIndex = (byteOffset == range.startIndex) ? utf8Offset : (utf8Offset - 1);

    while (byteOffset < range.stopIndex && sequenceLength != 0) {
      cur = AdvanceToNextUtf8Character(cur, sequenceLength);
      utf8Offset++;
      byteOffset += sequenceLength;
    }

    // we need to account for multi byte utf8, the |stopIndex| value should always point the the last byte of the range
    // which will put the utf8Offset one index ahead
    range.stopIndex = (byteOffset == range.stopIndex) ? utf8Offset : (utf8Offset - 1);
  }

  return ranges;
}

// allow only supported templates to be defined or else compile time errors
template std::vector<AutoModFlagsRange> ConvertUtf8RangesToByteRanges<AutoModFlagsRange>(
  const std::vector<AutoModFlagsRange>& utf8Ranges, const std::string& message);

template std::vector<TokenRange> ConvertUtf8RangesToByteRanges<TokenRange>(
  const std::vector<TokenRange>& utf8Ranges, const std::string& message);

template <typename T>
std::vector<T> ConvertUtf8RangesToByteRanges(const std::vector<T>& utf8Ranges, const std::string& message) {
  // This assumes that the ranges are sorted first by start index
  auto byteRanges = GetUtf8ToByteRanges(ToRangeBases(utf8Ranges), message);

  return FromRangeBases(utf8Ranges, byteRanges);
}

// allow only supported templates to be defined or else compile time errors
template std::vector<TokenRange> ConvertByteRangesToUtf8Ranges<TokenRange>(
  const std::vector<TokenRange>& utf8Ranges, const std::string& message);

template <typename T>
std::vector<T> ConvertByteRangesToUtf8Ranges(const std::vector<T>& byteRanges, const std::string& message) {
  // This assumes that the ranges are sorted first by start index
  auto utf8Ranges = GetByteToUtf8Ranges(ToRangeBases(byteRanges), message);

  return FromRangeBases(byteRanges, utf8Ranges);
}

/**
 * Converts EmoticonRange sets to TokenRange.  The index number will be the same order as |ranges|.  |messageLength| is
 * the length of the original message body equivlent to message.length()
 * The returned TokenRanges are sorted by startIndex
 */
std::vector<TokenRange> ConvertToTokenRanges(
  const std::map<std::string, std::vector<EmoteRange>>& emoticonCollections, size_t messageLength) {
  if (messageLength == 0) {
    return {};
  }

  std::vector<TokenRange> tokenRanges;
  int rangeIndex = 0;
  for (auto& collection : emoticonCollections) {
    std::string emoticonId = collection.first;
    const std::vector<EmoteRange>& emoteRanges = collection.second;

    for (const auto& emoteRange : emoteRanges) {
      if (static_cast<size_t>(emoteRange.startIndex) >= messageLength ||
          static_cast<size_t>(emoteRange.endIndex) >= messageLength) {
        continue;
      }

      TokenRange range;
      range.startIndex = emoteRange.startIndex;
      range.endIndex = emoteRange.endIndex;
      range.type = TOKEN_RANGE_TYPE_EMOTICON;
      range.rangeNum = rangeIndex++;
      range.emoticon.emoticonId = emoticonId;

      tokenRanges.emplace_back(range);
    }
  }

  std::sort(tokenRanges.begin(), tokenRanges.end(), tokenranges::SortTokenRangesByStartIndex);

  return tokenRanges;
}

bool SortTokenRangesByTypeThenRangeNum(const TokenRange& a, const TokenRange& b) {
  if (a.type < b.type) {
    return true;
  } else if (a.type > b.type) {
    return false;
  } else {
    return a.rangeNum < b.rangeNum;
  }
}

bool SortTokenRangesByStartIndex(const TokenRange& a, const TokenRange& b) {
  if (a.startIndex < b.startIndex) {
    return true;
  } else if (a.startIndex > b.startIndex) {
    return false;
  } else {
    return SortTokenRangesByTypeThenRangeNum(a, b);
  }
}

bool SortEmoticonRangesByEmoticonId(const TokenRange& a, const TokenRange& b) {
  if (a.emoticon.emoticonId == b.emoticon.emoticonId) {
    return SortTokenRangesByTypeThenRangeNum(a, b);
  }

  return CompareEmoticonId(a.emoticon.emoticonId, b.emoticon.emoticonId);
}

void RemoveOverlappingRanges(std::vector<TokenRange>& ranges) {
  if (ranges.size() <= 1) {
    return;
  }

  // This assumes that the ranges are sorted first by start index
  // Remove ranges that overlap, keeping the left-most range
  TokenRange current = ranges.front();

  for (auto iter = ranges.begin() + 1; iter != ranges.end();) {
    TokenRange& range = *iter;

    // Overlaps, remove it
    if (range.startIndex <= current.endIndex) {
      iter = ranges.erase(iter);
    }
    // No overlap, keep it
    else {
      current = range;
      iter++;
    }
  }
}

}  // namespace tokenranges
}  // namespace chat
}  // namespace ttv
