/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/internal/useremoticonsets.h"

#include "twitchsdk/chat/internal/chathelpers.h"
#include "twitchsdk/chat/internal/chatmessageparsing.h"
#include "twitchsdk/chat/internal/ircstring.h"
#include "twitchsdk/chat/internal/task/chatgetemoticonstask.h"
#include "twitchsdk/core/coreutilities.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/corejsonutil.h"
#include "twitchsdk/core/json/reader.h"
#include "twitchsdk/core/stringutilities.h"
#include "twitchsdk/core/user/user.h"

namespace {
const char* kLogger = "UserEmoticonSets";
const char* kTopicPrefix = "user-subscribe-events-v1.";

const uint64_t kFetchRetryJitterMilliseconds = 1000;
const uint64_t kFetchRetryMaxBackOff = 1000 * 60 * 2;  // 2 minutes
}  // namespace

ttv::chat::UserEmoticonSets::UserEmoticonSets(const std::shared_ptr<User>& user, const TokenizationOptions& options)
    : PubSubComponent(user),
      mPubSubTopic(kTopicPrefix + std::to_string(user->GetUserId())),
      mFetchRetryTimer(kFetchRetryMaxBackOff, kFetchRetryJitterMilliseconds),
      mTokenizationOptions(options),
      mEmoticonFetchInFlight(false) {
  AddTopic(mPubSubTopic);
}

std::string ttv::chat::UserEmoticonSets::GetLoggerName() const {
  return kLogger;
}

TTV_ErrorCode ttv::chat::UserEmoticonSets::Initialize() {
  ttv::trace::Message(kLogger, MessageLevel::Debug, "Initialize()");

  TTV_ErrorCode ec = PubSubComponent::Initialize();

  if (TTV_SUCCEEDED(ec)) {
    mFetchRetryTimer.ScheduleNextRetry();
  }

  return ec;
}

void ttv::chat::UserEmoticonSets::Update() {
  if (mState == State::Uninitialized) {
    return;
  }

  PubSubComponent::Update();

  if (mState == State::Initialized) {
    if (mFetchRetryTimer.IsRetrySet() && mFetchRetryTimer.CheckNextRetry()) {
      FetchUserEmoticonSets(true, nullptr);
    }
  }
}

TTV_ErrorCode ttv::chat::UserEmoticonSets::Shutdown() {
  ttv::trace::Message(kLogger, MessageLevel::Debug, "Shutdown()");

  TTV_ErrorCode ec = PubSubComponent::Shutdown();

  if (TTV_SUCCEEDED(ec)) {
    mFetchRetryTimer.Clear();
  }

  return ec;
}

TTV_ErrorCode ttv::chat::UserEmoticonSets::GetUserEmoticonSets(std::vector<EmoticonSet>& sets) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!mTokenizationOptions.emoticons) {
    return TTV_EC_FEATURE_DISABLED;
  }

  if (mUserEmoticonSets.empty()) {
    return TTV_EC_CHAT_EMOTICON_DATA_DOWNLOADING;
  }

  sets = mUserEmoticonSets;
  return TTV_EC_SUCCESS;
}

TTV_ErrorCode ttv::chat::UserEmoticonSets::FetchUserEmoticonSets(
  bool forceRefetch, const FetchEmoticonSetsCallback& callback) {
  if (mState != State::Initialized) {
    return TTV_EC_SHUT_DOWN;
  }

  if (!mTokenizationOptions.emoticons) {
    return TTV_EC_FEATURE_DISABLED;
  }

  if (!forceRefetch) {
    // Use cached values
    if (callback == nullptr) {
      return TTV_EC_INVALID_CALLBACK;
    }

    if (mUserEmoticonSets.empty()) {
      // We don't have any cached values yet, store callback to call later
      mFetchUserEmoticonSetsCallbacks.Push(callback);
    } else {
      callback(TTV_EC_SUCCESS, mUserEmoticonSets);
    }

    return TTV_EC_SUCCESS;
  }

  auto user = mUser.lock();
  if (user == nullptr || user->GetUserId() == 0) {
    return TTV_EC_NEED_TO_LOGIN;
  }
  auto oauthToken = user->GetOAuthToken();

  if (mEmoticonFetchInFlight) {
    if (callback != nullptr) {
      mFetchUserEmoticonSetsCallbacks.Push(callback);
    }

    return TTV_EC_SUCCESS;
  }

  auto task = std::make_shared<ChatGetEmoticonsTask>(user->GetUserId(), oauthToken->GetToken(),
    [this, user, oauthToken](ChatGetEmoticonsTask* source, TTV_ErrorCode ec, std::vector<EmoticonSet>&& result) {
      CompleteTask(source);
      mEmoticonFetchInFlight = false;

      if (TTV_SUCCEEDED(ec)) {
        mFetchRetryTimer.Clear();

        std::vector<EmoticonSet> fetchedEmoticonSets = std::move(result);

        // Sort by emoticon set ID in decreasing order for proper ordering, also to tell if emoticon sets have changed.
        std::sort(
          fetchedEmoticonSets.rbegin(), fetchedEmoticonSets.rend(), [](const EmoticonSet& lhs, const EmoticonSet& rhs) {
            // CompareEmoticonId only support less than comparison, must reverse post
            // sort, use reverse iterator or create new comparision funciton.
            return CompareEmoticonId(lhs.emoticonSetId, rhs.emoticonSetId);
          });

        for (auto& emoticonSet : fetchedEmoticonSets) {
          std::sort(emoticonSet.emoticons.begin(), emoticonSet.emoticons.end(),
            [](const Emoticon& lhs, const Emoticon& rhs) { return CompareEmoticonId(lhs.emoticonId, rhs.emoticonId); });
        }

        if (mUserEmoticonSets != fetchedEmoticonSets) {
          mUserEmoticonSets = fetchedEmoticonSets;
          mListener->OnUserEmoticonSetsChanged(user->GetUserId(), mUserEmoticonSets);
        }
      } else {
        if (ec == TTV_EC_AUTHENTICATION) {
          user->ReportOAuthTokenInvalid(oauthToken, ec);
        }

        mFetchRetryTimer.ScheduleNextRetry();
      }

      mFetchUserEmoticonSetsCallbacks.Flush(ec, mUserEmoticonSets);
    });

  TTV_ErrorCode ec = StartTask(task);

  if (TTV_SUCCEEDED(ec)) {
    mEmoticonFetchInFlight = true;

    if (callback != nullptr) {
      mFetchUserEmoticonSetsCallbacks.Push(callback);
    }
  }

  return ec;
}

void ttv::chat::UserEmoticonSets::OnTopicSubscribeStateChanged(
  const std::string& /*topic*/, PubSubClient::SubscribeState::Enum state, TTV_ErrorCode /*ec*/) {
  ttv::trace::Message(kLogger, MessageLevel::Debug, "UserEmoticonSets SubscribeStateChanged: %s",
    ((state == PubSubClient::SubscribeState::Subscribed) ? "subscribed" : "unsubscribed"));

  if (state == PubSubClient::SubscribeState::Subscribed) {
    mFetchRetryTimer.ScheduleNextRetry();
  }
}

void ttv::chat::UserEmoticonSets::OnTopicMessageReceived(const std::string& topic, const ttv::json::Value& jVal) {
  if (jVal.isNull() || !jVal.isObject()) {
    Log(MessageLevel::Error, "Invalid pubsub message json, dropping");
    return;
  }

  if (topic == mPubSubTopic) {
    UserId userId;

    if (!ParseUserId(jVal["user_id"], userId)) {
      Log(MessageLevel::Error, "Could not parse user ID, dropping");
      return;
    }

    auto user = mUser.lock();
    if (user == nullptr || user->GetUserId() == 0) {
      return;
    }

    if (userId == user->GetUserId()) {
      // Available emoticon sets have changed, re-fetch
      mFetchRetryTimer.ScheduleNextRetry();
    } else {
      Log(MessageLevel::Error, "User ID's do not match, dropping");
      return;
    }
  }
}
