/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#pragma once

#include "twitchsdk/chat/internal/pch.h"

#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichattransport.h"
#include "twitchsdk/chat/internal/useremoticonsets.h"
#include "twitchsdk/core/concurrentqueue.h"
#include "twitchsdk/core/httprequest.h"

#include <chrono>
#include <thread>

struct ChatTestApiState {
  ttv::chat::ChatAPI::State state;
  TTV_ErrorCode stateResult;

  ChatTestApiState() {
    state = ttv::chat::ChatAPI::State::Uninitialized;
    stateResult = TTV_EC_UNKNOWN_ERROR;
  }
};

class ChatTestManager : public ttv::chat::IChatChannelListener, public ttv::chat::IChatAPIListener {
 public:
  ChatTestManager();
  virtual ~ChatTestManager();

  void SetChatApi(std::shared_ptr<ttv::chat::ChatAPI> api) { mChatApi = api; }

  std::shared_ptr<ChatTestApiState> GetChatApiState();

  ttv::chat::ChatChannelState GetChatChannelStateForChannel(ttv::UserId userId, ttv::ChannelId channelId);
  ttv::chat::ChatChannelInfo GetChatChannelInfoFor(ttv::UserId userId, ttv::ChannelId channelId);
  ttv::chat::ChatChannelRestrictions GetChatChannelRestrictionsFor(ttv::UserId userId, ttv::ChannelId channelId);
  const std::vector<ttv::chat::EmoticonSet>& GetChatUserEmoticonSets(ttv::UserId userId);
  bool HasMessageList(
    const std::vector<ttv::chat::MessageInfo>& messageList, ttv::UserId userId, ttv::ChannelId channelId);
  bool AssertMessage(const ttv::chat::MessageInfo& actual, const ttv::chat::MessageInfo& expected);
  bool ChannelExists(ttv::UserId userId, ttv::ChannelId channelId);
  void FreeAllMessages();

  // IModuleListener Callbacks
 public:
  virtual void ModuleStateChanged(ttv::IModule* source, ttv::IModule::State state, TTV_ErrorCode ec) override;

  // IChatChannelListener Callbacks
 public:
  virtual void ChatChannelStateChanged(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::chat::ChatChannelState state, TTV_ErrorCode ec) override;
  virtual void ChatChannelInfoChanged(
    ttv::UserId userId, ttv::ChannelId channelId, const ttv::chat::ChatChannelInfo& info) override;
  virtual void ChatChannelRestrictionsChanged(
    ttv::UserId userId, ttv::ChannelId channelId, const ttv::chat::ChatChannelRestrictions& restrictions) override;
  virtual void ChatChannelLocalUserChanged(
    ttv::UserId userId, ttv::ChannelId channelId, const ttv::chat::ChatUserInfo& userInfo) override;
  virtual void ChatChannelMessagesReceived(
    ttv::UserId userId, ttv::ChannelId channelId, const std::vector<ttv::chat::LiveChatMessage>& messageList) override;
  virtual void ChatChannelSubscriptionNoticeReceived(
    ttv::UserId userId, ttv::ChannelId channelId, const ttv::chat::SubscriptionNotice& notice) override;
  virtual void ChatChannelFirstTimeChatterNoticeReceived(
    ttv::UserId userId, ttv::ChannelId channelId, const ttv::chat::FirstTimeChatterNotice& notice) override;
  virtual void ChatChannelRaidNoticeReceived(
    ttv::UserId userId, ttv::ChannelId channelId, const ttv::chat::RaidNotice& notice) override;
  virtual void ChatChannelUnraidNoticeReceived(
    ttv::UserId userId, ttv::ChannelId channelId, const ttv::chat::UnraidNotice& notice) override;
  virtual void ChatChannelGenericNoticeReceived(
    ttv::UserId userId, ttv::ChannelId channelId, const ttv::chat::GenericMessageNotice& notice) override;
  virtual void ChatChannelMessagesCleared(ttv::UserId userId, ttv::ChannelId channelId) override;
  virtual void ChatChannelUserMessagesCleared(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId clearUserId) override;
  virtual void ChatChannelMessageDeleted(ttv::UserId userId, ttv::ChannelId channelId, std::string&& messageId,
    std::string&& senderLoginName, std::string&& deletedMessageContent) override;
  virtual void ChatChannelModNoticeUserTimedOut(ttv::UserId userId, ttv::ChannelId channelId,
    ttv::chat::ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) override;
  virtual void ChatChannelModNoticeUserBanned(ttv::UserId userId, ttv::ChannelId channelId,
    ttv::chat::ModerationActionInfo&& modActionInfo, std::string&& reason) override;
  virtual void ChatChannelModNoticeUserUntimedOut(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::chat::ModerationActionInfo&& modActionInfo) override;
  virtual void ChatChannelModNoticeUserUnbanned(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::chat::ModerationActionInfo&& modActionInfo) override;
  virtual void ChatChannelModNoticeMessageDeleted(ttv::UserId userId, ttv::ChannelId channelId,
    ttv::chat::ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) override;
  virtual void ChatChannelModNoticeClearChat(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelModNoticeEmoteOnly(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelModNoticeEmoteOnlyOff(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelModNoticeFollowersOnly(ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId,
    std::string&& modName, uint32_t minimumFollowingDurationMinutes) override;
  virtual void ChatChannelModNoticeFollowersOnlyOff(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelModNoticeR9K(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelModNoticeR9KOff(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelModNoticeSlow(ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId,
    std::string&& modName, uint32_t slowModeDurationSeconds) override;
  virtual void ChatChannelModNoticeSlowOff(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelModNoticeSubsOnly(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelModNoticeSubsOnlyOff(
    ttv::UserId userId, ttv::ChannelId channelId, ttv::UserId modId, std::string&& modName) override;
  virtual void ChatChannelHostTargetChanged(
    ttv::UserId userId, ttv::ChannelId channelId, const std::string& targetChannelName, uint32_t numViewers) override;
  virtual void ChatChannelNoticeReceived(ttv::UserId userId, ttv::ChannelId channelId, const std::string& id,
    const std::map<std::string, std::string>& params) override;
  virtual void AutoModCaughtSentMessage(ttv::UserId userId, ttv::ChannelId channelId) override;
  virtual void AutoModDeniedSentMessage(ttv::UserId userId, ttv::ChannelId channelId) override;
  virtual void AutoModApprovedSentMessage(ttv::UserId userId, ttv::ChannelId channelId) override;
  virtual void AutoModCaughtMessageForMods(ttv::UserId userId, ttv::ChannelId channelId, std::string&& messageId,
    std::string&& message, ttv::UserId senderId, std::string&& senderName, std::string&& reason) override;
  virtual void AutoModMessageApprovedByMod(ttv::UserId userId, ttv::ChannelId channelId, std::string&& messageId,
    ttv::UserId moderatorId, std::string&& moderatorName) override;
  virtual void AutoModMessageDeniedByMod(ttv::UserId userId, ttv::ChannelId channelId, std::string&& messageId,
    ttv::UserId moderatorId, std::string&& moderatorName) override;
  virtual void AutoModDeniedSentCheer(ttv::UserId userId, ttv::ChannelId channelId) override;
  virtual void AutoModTimedOutSentCheer(ttv::UserId userId, ttv::ChannelId channelId) override;
  virtual void AutoModCaughtCheerForMods(ttv::UserId userId, ttv::ChannelId channelId, std::string&& messageId,
    std::string&& message, ttv::UserId senderId, std::string&& senderName, std::string&& reason) override;

  // IChatAPIListener Callbacks
 public:
  virtual void ChatUserEmoticonSetsChanged(
    ttv::UserId userId, const std::vector<ttv::chat::EmoticonSet>& sets) override;

 private:
  std::shared_ptr<ttv::chat::ChatAPI> mChatApi;
  std::shared_ptr<ChatTestApiState> mChatTestApiState;
  std::map<ttv::UserId, std::map<ttv::ChannelId, ttv::chat::ChatChannelState>> mChatChannelStateMap;
  std::map<ttv::UserId, std::map<ttv::ChannelId, ttv::chat::ChatChannelInfo>> mChatChannelInfoMap;
  std::map<ttv::UserId, std::map<ttv::ChannelId, ttv::chat::ChatChannelRestrictions>> mChatChannelRestrictionsMap;
  std::map<ttv::UserId, std::map<ttv::ChannelId, std::vector<ttv::chat::LiveChatMessage>>> mChatChannelMessages;
  std::map<ttv::UserId, std::vector<ttv::chat::EmoticonSet>> mChatUserEmoticonSetsMap;
};
