/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "twitchsdk/core/base64.h"

#include "gtest/gtest.h"

namespace {
class Base64Test : public ::testing::Test {};
}  // namespace

/*
import base64
encoded = base64.b64encode(b"hello")
print encoded
*/

TEST(Base64Test, Hello) {
  std::string helloStr = "hello";

  std::string result = ttv::Base64Encode(
    reinterpret_cast<const unsigned char*>(helloStr.c_str()), static_cast<unsigned int>(helloStr.length()));
  ASSERT_EQ(result, "aGVsbG8=");

  result = ttv::Base64Decode(result);
  ASSERT_EQ(helloStr, result);
}

TEST(Base64Test, EmptyString) {
  std::string emptyStr = "";

  std::string result = ttv::Base64Encode(
    reinterpret_cast<const unsigned char*>(emptyStr.c_str()), static_cast<unsigned int>(emptyStr.length()));
  ASSERT_EQ(result, "");

  result = ttv::Base64Decode(result);
  ASSERT_EQ(emptyStr, result);
}

TEST(Base64Test, Length4) {
  std::string originalStr = "1234";

  std::string result = ttv::Base64Encode(
    reinterpret_cast<const unsigned char*>(originalStr.c_str()), static_cast<unsigned int>(originalStr.length()));
  ASSERT_EQ(result, "MTIzNA==");

  result = ttv::Base64Decode(result);
  ASSERT_EQ(originalStr, result);
}

TEST(Base64Test, Length5) {
  std::string originalStr = "12345";

  std::string result = ttv::Base64Encode(
    reinterpret_cast<const unsigned char*>(originalStr.c_str()), static_cast<unsigned int>(originalStr.length()));
  ASSERT_EQ(result, "MTIzNDU=");

  result = ttv::Base64Decode(result);
  ASSERT_EQ(originalStr, result);
}

TEST(Base64Test, Length6) {
  std::string originalStr = "123456";

  std::string result = ttv::Base64Encode(
    reinterpret_cast<const unsigned char*>(originalStr.c_str()), static_cast<unsigned int>(originalStr.length()));
  ASSERT_EQ(result, "MTIzNDU2");

  result = ttv::Base64Decode(result);
  ASSERT_EQ(originalStr, result);
}

TEST(Base64Test, EncodeMultipleTimes) {
  /*
     import base64
     string = "hello"
     for i in range(10):
        string = base64.b64encode(string)

     print string
    */

  std::string originalStr = "hello";
  std::string result = originalStr;

  for (int i = 0; i < 10; i++) {
    result = ttv::Base64Encode(
      reinterpret_cast<const unsigned char*>(result.c_str()), static_cast<unsigned int>(result.length()));
  }
  ASSERT_EQ(result,
    "Vm0wd2VHUXhTWGhpUm1SWVYwZDRWRll3Wkc5WFZsbDNXa1JTVjJKR2JETlhhMUpUVmpGYWRHVkdXbFpOYWtFeFZtMTRZV014WkhWaVJtUlhUVEZLVFZac1ZtRldNVnBXVFZWV2FHVnFRVGs9");

  for (int i = 0; i < 10; i++) {
    result = ttv::Base64Decode(result);
  }

  ASSERT_EQ(originalStr, result);
}

TEST(Base64Test, MinutesWatched) {
  std::string originalStr =
    "[{\"event\":\"benchmark_complete_transition\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"destination\":\"home\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"is_app_launch\":true,\"language\":\"en-US\",\"logged_in\":true,\"login\":\"elisdev04\",\"lost_visibility\":false,\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"time\":1542169960,\"time_from_fetch\":7886,\"turbo\":false,\"user_id\":146738107}}\n,{\"event\":\"benchmark_custom_event\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"destination\":\"home\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"duration\":6917,\"is_app_launch\":true,\"key\":\"first-cue\",\"language\":\"en-US\",\"logged_in\":true,\"login\":\"elisdev04\",\"lost_visibility\":false,\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"time\":1542169961,\"turbo\":false,\"user_id\":146738107}}\n,{\"event\":\"player-loaded\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"language\":\"en-US\",\"logged_in\":true,\"login\":\"elisdev04\",\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"time\":1542169961,\"turbo\":false,\"user_id\":146738107}}\n,{\"event\":\"benchmark_custom_event\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"destination\":\"home\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"duration\":3082,\"is_app_launch\":true,\"key\":\"player-played\",\"language\":\"en-US\",\"logged_in\":true,\"login\":\"elisdev04\",\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"time\":1542169962,\"turbo\":false,\"user_id\":146738107}}\n,{\"event\":\"video-play\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"autoplayed\":true,\"backend\":\"player-core\",\"backend_version\":\"2.3.0-9b3f0ede\",\"broadcast_id\":\"31223340832\",\"channel\":\"relaxbeats\",\"channel_id\":153859298,\"chat_visible\":false,\"cluster\":\"sjc02\",\"content_mode\":\"live\",\"core_version\":\"2.3.0-9b3f0ede\",\"current_bitrate\":0,\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"fullscreen\":false,\"game\":\"Music & Performing Arts\",\"language\":\"en-US\",\"live\":true,\"logged_in\":true,\"login\":\"elisdev04\",\"manifest_cluster\":\"sjc02\",\"manifest_node\":\"video-weaver.sjc02\",\"master_download_time\":365,\"mature\":false,\"medium\":\"preview\",\"muted\":true,\"node\":\"video-edge-a66b64.sjc02\",\"partner\":true,\"platform\":\"xboxone\",\"play_session_id\":\"99277546098f4872a35a23359c3f9da\",\"player\":\"tv_preview\",\"playing_ad\":false,\"quality\":\"auto\",\"segment_download_time\":341,\"serving_id\":\"1bda6022d35942e4a003baa61d26e8c8\",\"stream_format\":\"360p30\",\"subscriber\":false,\"time\":1542169962.191436,\"time_since_load_start\":848,\"time_to_mp_master_manifest_request\":2.4036943990,\"time_to_mp_master_manifest_response\":2.5955889970,\"time_to_mp_variant_request\":2.7407865960,\"time_to_mp_variant_response\":2.7751194440,\"time_to_tv_page_visible\":0.6895599059999999,\"time_to_tv_user_request\":0.0,\"time_to_web_playback_started\":3.0823942070,\"transcoder_type\":\"2017TranscodeX264_V2\",\"turbo\":false,\"url\":\"https://www.twitch.tv/relaxbeats\",\"user_id\":146738107,\"variant_download_time\":140,\"vid_height\":360,\"vid_width\":576,\"video_buffer_size\":3.0462660,\"vod_timestamp\":null,\"volume\":0}}\n,{\"event\":\"abs_stream_format_change\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"autoplayed\":true,\"backend\":\"player-core\",\"backend_version\":\"2.3.0-9b3f0ede\",\"broadcast_id\":\"31223340832\",\"buffer_level\":3970,\"channel\":\"relaxbeats\",\"channel_id\":153859298,\"chat_visible\":false,\"cluster\":\"sjc02\",\"content_mode\":\"live\",\"core_version\":\"2.3.0-9b3f0ede\",\"current_bitrate\":630,\"current_stream_format_bitrate\":630,\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"estimated_bandwidth\":23256,\"fullscreen\":false,\"game\":\"Music & Performing Arts\",\"language\":\"en-US\",\"live\":true,\"logged_in\":true,\"login\":\"elisdev04\",\"manifest_cluster\":\"sjc02\",\"manifest_node\":\"video-weaver.sjc02\",\"mature\":false,\"medium\":\"preview\",\"muted\":true,\"node\":\"video-edge-a66b64.sjc02\",\"partner\":true,\"platform\":\"xboxone\",\"play_session_id\":\"99277546098f4872a35a23359c3f9da\",\"player\":\"tv_preview\",\"playing_ad\":false,\"quality\":\"auto\",\"serving_id\":\"1bda6022d35942e4a003baa61d26e8c8\",\"stream_format\":\"360p30\",\"subscriber\":false,\"time\":1542169962,\"transcoder_type\":\"2017TranscodeX264_V2\",\"turbo\":false,\"url\":\"https://www.twitch.tv/relaxbeats\",\"user_id\":146738107,\"vid_height\":360,\"vid_width\":576,\"video_buffer_size\":3.9706120,\"vod_timestamp\":null,\"volume\":0}}\n,{\"event\":\"ui_interaction\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"interaction\":\"click\",\"item_name\":\"theater_mode_button\",\"language\":\"en-US\",\"logged_in\":true,\"login\":\"elisdev04\",\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"screen_name\":\"home\",\"section_header\":\"streams_followed\",\"target_user_id\":153859298,\"time\":1542169964,\"turbo\":false,\"user_id\":146738107}}\n,{\"event\":\"video_ad_request_declined\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"channel\":\"relaxbeats\",\"channel_id\":153859298,\"chat_visible\":false,\"content_mode\":\"live\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"fullscreen\":true,\"game\":\"Music & Performing Arts\",\"language\":\"en-US\",\"live\":true,\"logged_in\":true,\"login\":\"elisdev04\",\"mature\":false,\"partner\":true,\"platform\":\"xboxone\",\"play_session_id\":\"99277546098f4872a35a23359c3f9da\",\"player\":\"xbox_one\",\"provider\":\"DFP\",\"reason_ratelimit\":true,\"roll_type\":\"preroll\",\"subscriber\":false,\"time\":1542169965,\"time_break\":30,\"turbo\":false,\"user_id\":146738107}}\n,{\"event\":\"pageview\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"language\":\"en-US\",\"location\":\"playback\",\"logged_in\":true,\"login\":\"elisdev04\",\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"time\":1542169965,\"turbo\":false,\"user_id\":146738107}}\n,{\"event\":\"screen_view\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"language\":\"en-US\",\"logged_in\":true,\"login\":\"elisdev04\",\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"screen_name\":\"theater_mode\",\"time\":1542169965,\"turbo\":false,\"user_id\":146738107,\"viewed_user_id\":153859298}}\n,{\"event\":\"pageview\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"channel_id\":153859298,\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"language\":\"en-US\",\"location\":\"theater_mode\",\"logged_in\":true,\"login\":\"elisdev04\",\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"time\":1542169965,\"turbo\":false,\"user_id\":146738107}}\n,{\"event\":\"video-play\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"player-core\",\"backend_version\":\"2.3.0-9b3f0ede\",\"broadcast_id\":\"31223340832\",\"channel\":\"relaxbeats\",\"channel_id\":153859298,\"chat_visible\":false,\"cluster\":\"sjc02\",\"content\":\"followed-live\",\"content_mode\":\"live\",\"core_version\":\"2.3.0-9b3f0ede\",\"current_bitrate\":630,\"current_fps\":19,\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"fullscreen\":true,\"game\":\"Music & Performing Arts\",\"language\":\"en-US\",\"live\":true,\"logged_in\":true,\"login\":\"elisdev04\",\"manifest_cluster\":\"sjc02\",\"manifest_node\":\"video-weaver.sjc02\",\"mature\":false,\"medium\":\"home\",\"muted\":false,\"node\":\"video-edge-a66b64.sjc02\",\"partner\":true,\"platform\":\"xboxone\",\"play_session_id\":\"292ba646011f43c296621e70c9f2203\",\"player\":\"xbox_one\",\"playing_ad\":false,\"quality\":\"auto\",\"serving_id\":\"1bda6022d35942e4a003baa61d26e8c8\",\"stream_format\":\"chunked\",\"subscriber\":false,\"time\":1542169965.415035,\"time_to_mp_master_manifest_request\":2.4036943990,\"time_to_mp_master_manifest_response\":2.5955889970,\"time_to_mp_variant_request\":5.3713801320,\"time_to_mp_variant_response\":5.4114782540,\"time_to_tv_page_visible\":0.6895599059999999,\"time_to_tv_user_request\":0.0,\"time_to_web_playback_started\":3.0823942070,\"transcoder_type\":\"2017TranscodeX264_V2\",\"turbo\":false,\"url\":\"https://www.twitch.tv/relaxbeats\",\"user_id\":146738107,\"vid_height\":700,\"vid_width\":1120,\"video_buffer_size\":7.0124260,\"vod_timestamp\":null,\"volume\":0}}\n,{\"event\":\"benchmark_custom_event\",\"properties\":{\"app_session_id\":\"a7bf6bd0-f350-2964-9735-8b13820bb9a2\",\"app_version\":\"2.2.3.test6\",\"backend\":\"mediaplayer\",\"backend_version\":\"2.3.0-9b3f0ede\",\"destination\":\"channel\",\"device_id\":\"4e7a440fed3f8bfb2a50dd81f07dcd1dfc68ebc6\",\"device_model\":\"Xbox One\",\"duration\":484,\"is_app_launch\":false,\"key\":\"chat-connected\",\"language\":\"en-US\",\"logged_in\":true,\"login\":\"elisdev04\",\"platform\":\"xboxone\",\"player\":\"xbox_one\",\"time\":1542169965,\"turbo\":false,\"user_id\":146738107}}\n]";
  std::string base64Str =
    "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";

  // encode then decode same string and see if it's the same
  std::string result = ttv::Base64Encode(
    reinterpret_cast<const unsigned char*>(originalStr.c_str()), static_cast<unsigned int>(originalStr.length()));
  result = ttv::Base64Decode(result);
  ASSERT_EQ(originalStr, result);

  // see if encode/decode is a 1 to 1 mapping
  result = ttv::Base64Decode(base64Str);
  ASSERT_EQ(result, originalStr);

  result = ttv::Base64Encode(
    reinterpret_cast<const unsigned char*>(originalStr.c_str()), static_cast<unsigned int>(originalStr.length()));
  ASSERT_EQ(result, base64Str);
}
