/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ibitslistener.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
class TestBitsListener : public ttv::chat::IBitsListener {
 public:
  using BitsReceivedCallback = std::function<void(const BitsReceivedEvent& bitsReceievedEvent)>;

  virtual void UserReceivedBits(const BitsReceivedEvent& bitsReceievedEvent) override {
    if (bitsReceivedCallback != nullptr) {
      bitsReceivedCallback(bitsReceievedEvent);
    }
  }

  virtual void UserSentBits(const BitsSentEvent& /*bitsSentEvent*/) override {}

  virtual void UserGainedBits(const uint32_t /*bitsBalance*/) override {}

  BitsReceivedCallback bitsReceivedCallback;
};

class BitsListenerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions::All());

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "bits_receiver";
    mUserInfo.displayName = "bits_receiver";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mBitsListener = std::make_shared<TestBitsListener>();

    mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
      .SetType(ttv::HTTP_POST_REQUEST)
      .SetStatusCode(200)
      .AddRequestRegex(std::regex("FetchChannelCheermotes"))
      .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
      .Done();
  }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestBitsListener> mBitsListener;
  std::shared_ptr<ChatAPI> mChatApi;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};
}  // namespace

TEST_F(BitsListenerTest, NewBits) {
  bool receivedCallback = false;
  mBitsListener->bitsReceivedCallback = [&receivedCallback](const BitsReceivedEvent& bitsReceivedEvent) {
    ASSERT_EQ(bitsReceivedEvent.channelName, "bits_receiver");
    ASSERT_EQ(bitsReceivedEvent.context, "cheer");
    ASSERT_EQ(bitsReceivedEvent.channelId, 12345);
    ASSERT_EQ(bitsReceivedEvent.bitsUsed, 5);
    ASSERT_EQ(bitsReceivedEvent.totalBitsUsed, 113);
    ASSERT_EQ(bitsReceivedEvent.badge.isNewBadgeLevel, false);
    ASSERT_EQ(bitsReceivedEvent.badge.previousLevel, 0);
    ASSERT_EQ(bitsReceivedEvent.badge.newLevel, 0);

    Timestamp expectedTimestamp;
    ASSERT_TRUE(ttv::RFC3339TimeToUnixTimestamp("2017-08-02T00:02:38.655Z", expectedTimestamp));
    ASSERT_EQ(bitsReceivedEvent.message.timestamp, expectedTimestamp);
    ASSERT_EQ(bitsReceivedEvent.message.userName, "guy_who_bits");
    ASSERT_EQ(bitsReceivedEvent.message.userId, 42265657);

    ASSERT_EQ(bitsReceivedEvent.message.tokens.size(), 9);

    for (int i = 0; i < 9; i += 2) {
      ASSERT_EQ(bitsReceivedEvent.message.tokens[i]->GetType(), MessageToken::Type::Bits);
      BitsToken* bitsToken = static_cast<BitsToken*>(bitsReceivedEvent.message.tokens[i].get());
      ASSERT_EQ(bitsToken->prefix, "cheer");
      ASSERT_EQ(bitsToken->numBits, 1);
    }

    for (int i = 1; i < 9; i += 2) {
      ASSERT_EQ(bitsReceivedEvent.message.tokens[i]->GetType(), MessageToken::Type::Text);
      TextToken* textToken = static_cast<TextToken*>(bitsReceivedEvent.message.tokens[i].get());
      ASSERT_EQ(textToken->text, " ");
    }

    receivedCallback = true;
  };

  std::shared_ptr<IBitsStatus> bitsStatus;
  TTV_ErrorCode ec = mChatApi->CreateBitsStatus(12345, mBitsListener, bitsStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(10000, [bitsStatus]() { return bitsStatus->IsReady(); }, GetDefaultUpdateFunc()));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("channel-bits-events-v1.12345"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("channel-bits-events-v1.12345",
    "{\"data\":{\"user_name\":\"guy_who_bits\",\"channel_name\":\"bits_receiver\",\"user_id\":\"42265657\",\"channel_id\":\"12345\",\"time\":\"2017-08-02T00:02:38.655Z\",\"chat_message\":\"cheer1 cheer1 cheer1 cheer1 cheer1\",\"bits_used\":5,\"total_bits_used\":113,\"context\":\"cheer\",\"badge_entitlement\":null},\"version\":\"1.0\",\"message_type\":\"bits_event\",\"message_id\":\"4f6b1f8f-5495-58ef-a713-5d6f45ae0aa9\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(BitsListenerTest, NewBitsBadgeEntitlement) {
  bool receivedCallback = false;
  mBitsListener->bitsReceivedCallback = [&receivedCallback](const BitsReceivedEvent& bitsReceivedEvent) {
    ASSERT_EQ(bitsReceivedEvent.channelName, "bits_receiver");
    ASSERT_EQ(bitsReceivedEvent.context, "cheer");
    ASSERT_EQ(bitsReceivedEvent.channelId, 12345);
    ASSERT_EQ(bitsReceivedEvent.bitsUsed, 102);
    ASSERT_EQ(bitsReceivedEvent.totalBitsUsed, 108);
    ASSERT_EQ(bitsReceivedEvent.badge.isNewBadgeLevel, true);
    ASSERT_EQ(bitsReceivedEvent.badge.previousLevel, 1);
    ASSERT_EQ(bitsReceivedEvent.badge.newLevel, 100);

    Timestamp expectedTimestamp;
    ASSERT_TRUE(ttv::RFC3339TimeToUnixTimestamp("2017-08-02T00:01:12.381Z", expectedTimestamp));
    ASSERT_EQ(bitsReceivedEvent.message.timestamp, expectedTimestamp);
    ASSERT_EQ(bitsReceivedEvent.message.userName, "guy_who_bits");
    ASSERT_EQ(bitsReceivedEvent.message.userId, 42265657);

    ASSERT_EQ(bitsReceivedEvent.message.tokens.size(), 7);

    ASSERT_EQ(bitsReceivedEvent.message.tokens[0]->GetType(), MessageToken::Type::Text);
    TextToken* textToken = static_cast<TextToken*>(bitsReceivedEvent.message.tokens[0].get());
    ASSERT_EQ(textToken->text, "Chee ");

    ASSERT_EQ(bitsReceivedEvent.message.tokens[1]->GetType(), MessageToken::Type::Bits);
    BitsToken* bitsToken = static_cast<BitsToken*>(bitsReceivedEvent.message.tokens[1].get());
    ASSERT_EQ(bitsToken->prefix, "cheer");
    ASSERT_EQ(bitsToken->numBits, 1);

    ASSERT_EQ(bitsReceivedEvent.message.tokens[2]->GetType(), MessageToken::Type::Text);
    textToken = static_cast<TextToken*>(bitsReceivedEvent.message.tokens[2].get());
    ASSERT_EQ(textToken->text, " this is cool Cheer1adf ");

    ASSERT_EQ(bitsReceivedEvent.message.tokens[3]->GetType(), MessageToken::Type::Bits);
    bitsToken = static_cast<BitsToken*>(bitsReceivedEvent.message.tokens[3].get());
    ASSERT_EQ(bitsToken->prefix, "cheer");
    ASSERT_EQ(bitsToken->numBits, 100);

    ASSERT_EQ(bitsReceivedEvent.message.tokens[4]->GetType(), MessageToken::Type::Text);
    textToken = static_cast<TextToken*>(bitsReceivedEvent.message.tokens[4].get());
    ASSERT_EQ(textToken->text, " \"\"adfadf ");

    ASSERT_EQ(bitsReceivedEvent.message.tokens[5]->GetType(), MessageToken::Type::Bits);
    bitsToken = static_cast<BitsToken*>(bitsReceivedEvent.message.tokens[5].get());
    ASSERT_EQ(bitsToken->prefix, "cheer");
    ASSERT_EQ(bitsToken->numBits, 1);

    ASSERT_EQ(bitsReceivedEvent.message.tokens[6]->GetType(), MessageToken::Type::Text);
    textToken = static_cast<TextToken*>(bitsReceivedEvent.message.tokens[6].get());
    ASSERT_EQ(textToken->text, " Chee1");

    receivedCallback = true;
  };

  std::shared_ptr<IBitsStatus> bitsStatus;
  TTV_ErrorCode ec = mChatApi->CreateBitsStatus(12345, mBitsListener, bitsStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(10000, [bitsStatus]() { return bitsStatus->IsReady(); }, GetDefaultUpdateFunc()));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("channel-bits-events-v1.12345"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("channel-bits-events-v1.12345",
    "{\"data\":{\"user_name\":\"guy_who_bits\",\"channel_name\":\"bits_receiver\",\"user_id\":\"42265657\",\"channel_id\":\"12345\",\"time\":\"2017-08-02T00:01:12.381Z\",\"chat_message\":\"Chee cheer1 this is cool Cheer1adf cheer100 \\\"\\\"adfadf cheer1 Chee1\",\"bits_used\":102,\"total_bits_used\":108,\"context\":\"cheer\",\"badge_entitlement\":{\"new_version\":100,\"previous_version\":1}},\"version\":\"1.0\",\"message_type\":\"bits_event\",\"message_id\":\"ba52a135-7a0f-562b-95d9-156dcfa90383\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}
