/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ifollowerslistener.h"
#include "twitchsdk/core/coreapi.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
class TestFollowersListener : public ttv::chat::IFollowersListener {
 public:
  using NewFollowerCallback = std::function<void(const FollowerAddedEvent& followerAddedEvent)>;

  virtual void NewFollowerAdded(const FollowerAddedEvent& followerAddedEvent) override {
    if (newFollowerCallback != nullptr) {
      newFollowerCallback(followerAddedEvent);
    }
  }

  NewFollowerCallback newFollowerCallback;
};

class FollowersListenerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions());

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "winston";
    mUserInfo.displayName = "Winston";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mFollowersListener = std::make_shared<TestFollowersListener>();
  }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestFollowersListener> mFollowersListener;
  std::shared_ptr<ChatAPI> mChatApi;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};
}  // namespace

TEST_F(FollowersListenerTest, NewFollower) {
  bool receivedCallback = false;
  mFollowersListener->newFollowerCallback = [&receivedCallback](const FollowerAddedEvent& followerAddedEvent) {
    ASSERT_EQ(followerAddedEvent.userId, 157328038);
    ASSERT_EQ(followerAddedEvent.userName, "test_user_name");
    ASSERT_EQ(followerAddedEvent.displayName, "test_display_name");

    receivedCallback = true;
  };

  std::shared_ptr<IFollowersStatus> followersStatus;
  TTV_ErrorCode ec = mChatApi->CreateFollowersStatus(12345, 12345, mFollowersListener, followersStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(
    1000, [this]() { return mPubSubTestUtility.IsSubscribedToTopic("following.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("following.12345",
    "{\"display_name\":\"test_display_name\",\"username\":\"test_user_name\",\"user_id\":\"157328038\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}
