/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ifollowinglistener.h"
#include "twitchsdk/core/coreapi.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
class TestFollowingListener : public ttv::chat::IFollowingListener {
 public:
  using FollowedChannelCallback = std::function<void(UserId userId, ChannelId channelId)>;
  using UnfollowedChannelCallback = std::function<void(UserId userId, ChannelId channelId)>;

  virtual void FollowedChannel(UserId userId, ChannelId channelId) override {
    if (followedChannelCallback != nullptr) {
      followedChannelCallback(userId, channelId);
    }
  }

  virtual void UnfollowedChannel(UserId userId, ChannelId channelId) override {
    if (unfollowedChannelCallback != nullptr) {
      unfollowedChannelCallback(userId, channelId);
    }
  }

  FollowedChannelCallback followedChannelCallback;
  UnfollowedChannelCallback unfollowedChannelCallback;
};

class FollowingListenerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions());

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "winston";
    mUserInfo.displayName = "Winston";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mFollowingListener = std::make_shared<TestFollowingListener>();
  }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestFollowingListener> mFollowingListener;
  std::shared_ptr<ChatAPI> mChatApi;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};
}  // namespace

TEST_F(FollowingListenerTest, Followed) {
  bool receivedCallback = false;
  mFollowingListener->followedChannelCallback = [this, &receivedCallback](UserId userId, ChannelId channelId) {
    ASSERT_EQ(userId, mUserInfo.userId);
    ASSERT_EQ(channelId, 157328038);

    receivedCallback = true;
  };

  std::shared_ptr<IFollowingStatus> followingStatus;
  TTV_ErrorCode ec = mChatApi->CreateFollowingStatus(12345, mFollowingListener, followingStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(
    1000, [this]() { return mPubSubTestUtility.IsSubscribedToTopic("follows.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage(
    "follows.12345", "{\"type\":\"user-followed\",\"target_user_id\":\"157328038\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(FollowingListenerTest, Unfollowed) {
  bool receivedCallback = false;
  mFollowingListener->unfollowedChannelCallback = [this, &receivedCallback](UserId userId, ChannelId channelId) {
    ASSERT_EQ(userId, mUserInfo.userId);
    ASSERT_EQ(channelId, 157328038);

    receivedCallback = true;
  };

  std::shared_ptr<IFollowingStatus> followingStatus;
  TTV_ErrorCode ec = mChatApi->CreateFollowingStatus(12345, mFollowingListener, followingStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(
    1000, [this]() { return mPubSubTestUtility.IsSubscribedToTopic("follows.12345"); }, GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage(
    "follows.12345", "{\"type\":\"user-unfollowed\",\"target_user_id\":\"157328038\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}
