/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/isubscriberslistener.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
class TestSubscribersListener : public ttv::chat::ISubscribersListener {
 public:
  using NewSubscriberCallback = std::function<void(const SubscriberAddedEvent& subscriberAddedEvent)>;

  virtual void NewSubscriberAdded(const SubscriberAddedEvent& subscriberAddedEvent) override {
    if (newSubscriberCallback != nullptr) {
      newSubscriberCallback(subscriberAddedEvent);
    }
  }

  NewSubscriberCallback newSubscriberCallback;
};

class SubscribersListenerTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions::All());

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "subs_receiver";
    mUserInfo.displayName = "subs_receiver";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    mSubscribersListener = std::make_shared<TestSubscribersListener>();
  }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<TestSubscribersListener> mSubscribersListener;
  std::shared_ptr<ChatAPI> mChatApi;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};
}  // namespace

TEST_F(SubscribersListenerTest, NewSubscription) {
  bool receivedCallback = false;
  mSubscribersListener->newSubscriberCallback = [&receivedCallback](const SubscriberAddedEvent& subscriberAddedEvent) {
    ASSERT_EQ(subscriberAddedEvent.userName, "guy_who_subs");
    ASSERT_EQ(subscriberAddedEvent.displayName, "test_name");
    ASSERT_EQ(subscriberAddedEvent.channelName, "subs_receiver");
    ASSERT_EQ(subscriberAddedEvent.userId, 157328038);
    ASSERT_EQ(subscriberAddedEvent.channelId, 12345);

    Timestamp expectedTimestamp;
    ASSERT_TRUE(ttv::RFC3339TimeToUnixTimestamp("2017-08-01T23:42:14.486989674Z", expectedTimestamp));
    ASSERT_EQ(subscriberAddedEvent.timestamp, expectedTimestamp);

    ASSERT_EQ(subscriberAddedEvent.subNotice.type, SubscriptionNotice::Type::Sub);
    ASSERT_EQ(subscriberAddedEvent.subNotice.plan, SubscriptionNotice::Plan::Prime);
    ASSERT_EQ(subscriberAddedEvent.subNotice.subStreakMonthCount, 1);
    ASSERT_EQ(subscriberAddedEvent.subNotice.subCumulativeMonthCount, 2);
    ASSERT_EQ(subscriberAddedEvent.subNotice.planDisplayName, "QA Test Subscription");
    ASSERT_EQ(subscriberAddedEvent.subNotice.systemMessage, "");
    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens.size(), 0);

    receivedCallback = true;
  };

  std::shared_ptr<ISubscribersStatus> subscribersStatus;
  TTV_ErrorCode ec = mChatApi->CreateSubscribersStatus(12345, mSubscribersListener, subscribersStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("channel-subscribe-events-v1.12345"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("channel-subscribe-events-v1.12345",
    "{\"user_name\":\"guy_who_subs\",\"display_name\":\"test_name\",\"channel_name\":\"subs_receiver\",\"user_id\":\"157328038\",\"channel_id\":\"12345\",\"time\":\"2017-08-01T23:42:14.486989674Z\",\"sub_message\":{\"message\":\"\",\"emotes\":null},\"sub_plan\":\"Prime\",\"sub_plan_name\":\"QA Test Subscription\",\"streak_months\":1,\"cumulative_months\":2,\"context\":\"sub\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(SubscribersListenerTest, ResubWithChatMessage) {
  bool receivedCallback = false;
  mSubscribersListener->newSubscriberCallback = [&receivedCallback](const SubscriberAddedEvent& subscriberAddedEvent) {
    ASSERT_EQ(subscriberAddedEvent.userName, "guy_who_subs");
    ASSERT_EQ(subscriberAddedEvent.displayName, "test_name");
    ASSERT_EQ(subscriberAddedEvent.channelName, "subs_receiver");
    ASSERT_EQ(subscriberAddedEvent.userId, 157328038);
    ASSERT_EQ(subscriberAddedEvent.channelId, 12345);

    Timestamp expectedTimestamp;
    ASSERT_TRUE(ttv::RFC3339TimeToUnixTimestamp("2017-08-01T23:42:14.486989674Z", expectedTimestamp));
    ASSERT_EQ(subscriberAddedEvent.timestamp, expectedTimestamp);

    ASSERT_EQ(subscriberAddedEvent.subNotice.type, SubscriptionNotice::Type::Resub);
    ASSERT_EQ(subscriberAddedEvent.subNotice.plan, SubscriptionNotice::Plan::Sub1000);
    ASSERT_EQ(subscriberAddedEvent.subNotice.subStreakMonthCount, 5);
    ASSERT_EQ(subscriberAddedEvent.subNotice.subCumulativeMonthCount, 10);
    ASSERT_EQ(subscriberAddedEvent.subNotice.planDisplayName, "QA Test Subscription");
    ASSERT_EQ(subscriberAddedEvent.subNotice.systemMessage, "");
    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens.size(), 12);

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[0]->GetType(), MessageToken::Type::Emoticon);
    EmoticonToken* emoticonToken =
      static_cast<EmoticonToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[0].get());
    ASSERT_EQ(emoticonToken->emoticonText, "KappaHD");
    ASSERT_EQ(emoticonToken->emoticonId, "2867");

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[1]->GetType(), MessageToken::Type::Text);
    TextToken* textToken = static_cast<TextToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[1].get());
    ASSERT_EQ(textToken->text, " ");

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[2]->GetType(), MessageToken::Type::Emoticon);
    emoticonToken = static_cast<EmoticonToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[2].get());
    ASSERT_EQ(emoticonToken->emoticonText, "VoHiYo");
    ASSERT_EQ(emoticonToken->emoticonId, "81274");

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[3]->GetType(), MessageToken::Type::Text);
    textToken = static_cast<TextToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[3].get());
    ASSERT_EQ(textToken->text, " A Twitch baby is born! ");

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[4]->GetType(), MessageToken::Type::Emoticon);
    emoticonToken = static_cast<EmoticonToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[4].get());
    ASSERT_EQ(emoticonToken->emoticonText, "KappaHD");
    ASSERT_EQ(emoticonToken->emoticonId, "2867_BW");

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[5]->GetType(), MessageToken::Type::Text);
    textToken = static_cast<TextToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[5].get());
    ASSERT_EQ(textToken->text, ". Thanks ");

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[6]->GetType(), MessageToken::Type::Mention);
    MentionToken* mentionToken =
      static_cast<MentionToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[6].get());
    ASSERT_EQ(mentionToken->userName, "subs_receiver");
    ASSERT_EQ(mentionToken->text, "subs_receiver");
    ASSERT_EQ(mentionToken->isLocalUser, true);

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[7]->GetType(), MessageToken::Type::Text);
    textToken = static_cast<TextToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[7].get());
    ASSERT_EQ(textToken->text, " (from ");

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[8]->GetType(), MessageToken::Type::Mention);
    mentionToken = static_cast<MentionToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[8].get());
    ASSERT_EQ(mentionToken->userName, "another_user");
    ASSERT_EQ(mentionToken->text, "@another_user");
    ASSERT_EQ(mentionToken->isLocalUser, false);

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[9]->GetType(), MessageToken::Type::Text);
    textToken = static_cast<TextToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[9].get());
    ASSERT_EQ(textToken->text, " - ");

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[10]->GetType(), MessageToken::Type::Url);
    UrlToken* urlToken = static_cast<UrlToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[10].get());
    ASSERT_EQ(urlToken->url, "http://twitch.tv");
    ASSERT_EQ(urlToken->hidden, false);

    ASSERT_EQ(subscriberAddedEvent.subNotice.userMessage->tokens[11]->GetType(), MessageToken::Type::Text);
    textToken = static_cast<TextToken*>(subscriberAddedEvent.subNotice.userMessage->tokens[11].get());
    ASSERT_EQ(textToken->text, " ).");

    receivedCallback = true;
  };

  std::shared_ptr<ISubscribersStatus> subscribersStatus;
  TTV_ErrorCode ec = mChatApi->CreateSubscribersStatus(12345, mSubscribersListener, subscribersStatus);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("channel-subscribe-events-v1.12345"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("channel-subscribe-events-v1.12345",
    "{\"user_name\":\"guy_who_subs\",\"display_name\":\"test_name\",\"channel_name\":\"subs_receiver\",\"user_id\":\"157328038\",\"channel_id\":\"12345\",\"time\":\"2017-08-01T23:42:14.486989674Z\",\"sub_message\":{\"message\":\"KappaHD VoHiYo A Twitch baby is born! KappaHD. Thanks subs_receiver (from @another_user - http://twitch.tv ).\",\"emotes\":[{\"start\":0,\"end\":6,\"id\":2867},{\"start\":8,\"end\":13,\"id\":81274},{\"start\":38,\"end\":44,\"id\":\"2867_BW\"}]},\"sub_plan\":\"1000\",\"sub_plan_name\":\"QA Test Subscription\",\"streak_months\":5,\"cumulative_months\":10,\"context\":\"resub\"}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}
