#include "chattestmanager.h"
#include "fixtures/chatapitest.h"
#include "testsystemclock.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichannelchatroommanager.h"
#include "twitchsdk/chat/ichannelchatroommanagerlistener.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/socket.h"
#include "twitchsdk/core/stringutilities.h"

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::chat::test;
using namespace ttv::test;

namespace {
const std::string kDefaultRoomId = "aabbccdd";
const UserId kDefaultUserId = 12826;

class TestChannelChatRoomManagerListener : public IChannelChatRoomManagerListener {
 public:
  void PurgeMessages(UserId /*userId*/, ChannelId /*channelId*/, Timestamp /*purgeAfter*/) override {}

  void RoomCreated(ChannelId /*ownerId*/, ChatRoomInfo&& /*roomInfo*/) override {}

  void RoomDeleted(ChannelId /*ownerId*/, ChatRoomInfo&& /*roomInfo*/) override {}
};

class ChannelChatRoomManagerTest : public ChatApiTest {
 public:
 protected:
  void LogIn(UserId userId) {
    TestParams testParams;
    testParams.userInfo.userId = userId;
    testParams.userInfo.userName = "valid_login";
    testParams.userInfo.displayName = "Display Name";
    testParams.oauth = "valid_oauth";

    TTV_ErrorCode ec = ChatApiTest::LogIn(testParams);
    EXPECT_EQ(TTV_EC_SUCCESS, ec);
  }

  void CreateChannelChatRoomManager(UserId userId) {
    mListener = std::make_shared<TestChannelChatRoomManagerListener>();
    ASSERT_TRUE(TTV_SUCCEEDED(
      mChatApi->CreateChannelChatRoomManager(userId, kDefaultUserId, mListener, mChannelChatRoomManager)));
  }

 protected:
  std::shared_ptr<ttv::chat::IChannelChatRoomManager> mChannelChatRoomManager;
  std::shared_ptr<TestChannelChatRoomManagerListener> mListener;
};
}  // namespace

TEST_F(ChannelChatRoomManagerTest, AddNewChatRoom) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_create.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateChannelChatRoomManager(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };

  RoomRolePermissions permissions(RoomRole::Everyone, RoomRole::Moderator);

  TTV_ErrorCode ec = mChannelChatRoomManager->AddNewChatRoom("newroom", "this is a topic", permissions,
    [&callbackCalled](TTV_ErrorCode ec, CreateRoomError&& error, ChatRoomInfo&& info) {
      callbackCalled = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::SUCCESS);

      ASSERT_EQ(info.id, "2cf8d794-46bd-4c29-8e6e-c602e1becaa9");
      ASSERT_EQ(info.name, "newroom");
      ASSERT_EQ(info.topic, "this is a topic");
      ASSERT_EQ(info.rolePermissions.read, RoomRole::Everyone);
      ASSERT_EQ(info.rolePermissions.send, RoomRole::Moderator);
      ASSERT_TRUE(info.view.isMuted);
      ASSERT_TRUE(info.view.isUnread);
      ASSERT_FALSE(info.view.isArchived);
      ASSERT_TRUE(info.view.permissions.readMessages);
      ASSERT_TRUE(info.view.permissions.sendMessages);
      ASSERT_TRUE(info.view.permissions.moderate);
      ASSERT_EQ(info.view.unreadMentionCount, 1);
      ASSERT_EQ(info.owner.userName, "twitch");
      ASSERT_EQ(info.owner.displayName, "Twitch");
      ASSERT_EQ(info.owner.bio, "The company.");
      ASSERT_EQ(info.owner.logoImageUrl,
        "https://static-cdn.jtvnw.net/user-default-pictures/27103734-3cda-44d6-a384-f2ab71e4bb85-profile_image-300x300.jpg");
      ASSERT_EQ(info.owner.userId, 12826);
      ASSERT_EQ(info.modes.slowModeDurationSeconds, 60);
      ASSERT_TRUE(info.modes.r9kMode);
      ASSERT_FALSE(info.modes.emotesOnlyMode);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChannelChatRoomManagerTest, AddNewChatRoom_Error) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  MockResponse& response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                             .SetType(HTTP_POST_REQUEST)
                             .SetStatusCode(200)
                             .SetResponseBodyFromFile("chat/chatroom_create_error.json");

  LogIn(kDefaultUserId);
  CreateChannelChatRoomManager(kDefaultUserId);

  RoomRolePermissions permissions(RoomRole::Everyone, RoomRole::Moderator);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mChannelChatRoomManager->AddNewChatRoom("no", "this is a topic", permissions,
    [&callbackCalled](TTV_ErrorCode ec, CreateRoomError&& error, ChatRoomInfo&& /*info*/) {
      callbackCalled = true;
      ASSERT_FALSE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::NAME_LENGTH_INVALID);
      ASSERT_EQ(error.maxLength, 25);
      ASSERT_EQ(error.minLength, 3);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());

  response.SetResponseBodyFromFile("chat/chatroom_create_error2.json");
  callbackCalled = false;
  ec = mChannelChatRoomManager->AddNewChatRoom("room", "too many rooms", permissions,
    [&callbackCalled](TTV_ErrorCode ec, CreateRoomError&& error, ChatRoomInfo&& /*info*/) {
      callbackCalled = true;
      ASSERT_FALSE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(error.code, GraphQLErrorCode::MAX_ROOMS_LIMIT_EXCEEDED);
      ASSERT_EQ(error.maxAllowedRooms, 3);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChannelChatRoomManagerTest, JoinChatRooms) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_join.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateChannelChatRoomManager(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mChannelChatRoomManager->JoinChatRooms([&callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChannelChatRoomManagerTest, LeaveChatRooms) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_leave.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateChannelChatRoomManager(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mChannelChatRoomManager->LeaveChatRooms([&callbackCalled](TTV_ErrorCode ec) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}

TEST_F(ChannelChatRoomManagerTest, FetchChatRoomsInfos) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatroom_fetchinfos.json")
    .Done();

  LogIn(kDefaultUserId);
  CreateChannelChatRoomManager(kDefaultUserId);

  bool callbackCalled = false;
  auto waitForCallbackFunc = [&callbackCalled]() { return callbackCalled; };
  TTV_ErrorCode ec = mChannelChatRoomManager->FetchChatRoomsInfo([&callbackCalled](TTV_ErrorCode ec,
                                                                   std::vector<ChatRoomInfo>&& infos) {
    callbackCalled = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    ASSERT_EQ(infos.size(), 3);

    ChatRoomInfo info = infos[0];
    ASSERT_EQ(info.id, "964786ee-f6e1-4458-b0d0-f1c58449649f");
    ASSERT_EQ(info.name, "room1");
    ASSERT_EQ(info.topic, "topic 1");
    ASSERT_EQ(info.rolePermissions.read, RoomRole::Everyone);
    ASSERT_EQ(info.rolePermissions.send, RoomRole::Moderator);
    ASSERT_TRUE(info.view.isMuted);
    ASSERT_TRUE(info.view.isUnread);
    ASSERT_FALSE(info.view.isArchived);
    ASSERT_TRUE(info.view.permissions.readMessages);
    ASSERT_TRUE(info.view.permissions.sendMessages);
    ASSERT_TRUE(info.view.permissions.moderate);
    ASSERT_EQ(info.view.unreadMentionCount, 1);
    ASSERT_EQ(info.owner.userName, "twitch");
    ASSERT_EQ(info.owner.displayName, "Twitch");
    ASSERT_EQ(info.owner.bio, "The company.");
    ASSERT_EQ(info.owner.logoImageUrl,
      "https://static-cdn.jtvnw.net/user-default-pictures/27103734-3cda-44d6-a384-f2ab71e4bb85-profile_image-300x300.jpg");
    ASSERT_EQ(info.owner.userId, 12826);
    ASSERT_EQ(info.modes.slowModeDurationSeconds, 20);
    ASSERT_TRUE(info.modes.r9kMode);
    ASSERT_FALSE(info.modes.emotesOnlyMode);

    info = infos[1];
    ASSERT_EQ(info.id, "6a2a2a49-b781-4d10-9caa-3e9a852ebef7");
    ASSERT_EQ(info.name, "room2");
    ASSERT_EQ(info.topic, "topic 2");
    ASSERT_EQ(info.rolePermissions.read, RoomRole::Subscriber);
    ASSERT_EQ(info.rolePermissions.send, RoomRole::Broadcaster);
    ASSERT_FALSE(info.view.isMuted);
    ASSERT_FALSE(info.view.isUnread);
    ASSERT_FALSE(info.view.isArchived);
    ASSERT_FALSE(info.view.permissions.readMessages);
    ASSERT_FALSE(info.view.permissions.sendMessages);
    ASSERT_FALSE(info.view.permissions.moderate);
    ASSERT_EQ(info.view.unreadMentionCount, 0);
    ASSERT_EQ(info.modes.slowModeDurationSeconds, 0);
    ASSERT_FALSE(info.modes.r9kMode);
    ASSERT_FALSE(info.modes.emotesOnlyMode);

    info = infos[2];
    ASSERT_EQ(info.id, "0dc55aa5-cfb1-4cc2-aa2d-371bcd2b1c6b");
    ASSERT_EQ(info.name, "room3");
    ASSERT_EQ(info.topic, "topic 3");
    ASSERT_EQ(info.rolePermissions.read, RoomRole::Everyone);
    ASSERT_EQ(info.rolePermissions.send, RoomRole::Broadcaster);
    ASSERT_FALSE(info.view.isMuted);
    ASSERT_FALSE(info.view.isUnread);
    ASSERT_FALSE(info.view.isArchived);
    ASSERT_FALSE(info.view.permissions.readMessages);
    ASSERT_FALSE(info.view.permissions.sendMessages);
    ASSERT_FALSE(info.view.permissions.moderate);
    ASSERT_EQ(info.view.unreadMentionCount, 0);
    ASSERT_EQ(info.modes.slowModeDurationSeconds, 0);
    ASSERT_FALSE(info.modes.r9kMode);
    ASSERT_FALSE(info.modes.emotesOnlyMode);
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_TRUE(WaitUntilResultWithPollTask(250, waitForCallbackFunc, GetDefaultUpdateFunc()));
  ASSERT_TRUE(waitForCallbackFunc());
}
