/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "chattestmanager.h"
#include "fixtures/chatapitest.h"
#include "testchatobjectfactory.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichatobjectfactory.h"
#include "twitchsdk/core/httprequestutils.h"
#include "twitchsdk/core/json/value.h"
#include "twitchsdk/core/socket.h"

#include <thread>

#include "gtest/gtest.h"

using namespace ttv::chat;
using namespace ttv::chat::test;
using namespace ttv::test;

namespace {
std::string ConcatValue(const std::string& prefix, int value) {
  return prefix + std::to_string(value);
}

class ChatUserThreadsProxyTester : public ttv::chat::IChatUserThreadsListener {
 public:
  ChatUserThreadsProxyTester() = default;
  virtual ~ChatUserThreadsProxyTester() = default;

  virtual void ChatThreadRealtimeMessageReceived(
    ttv::UserId userId, const std::string& threadId, const WhisperMessage& messageList) override {
    if (mChatThreadRealtimeMessageReceived) {
      mChatThreadRealtimeMessageReceived(userId, threadId, messageList);
    }
  }
  std::function<void(ttv::UserId userId, const std::string& threadId, const WhisperMessage& messageList)>
    mChatThreadRealtimeMessageReceived;
};

}  // namespace

TEST_F(ChatApiTest, InitalizationTestsSuccess) {
  TestChatApiInitializationCallback(TokenizationOptions());
}

TEST_F(ChatApiTest, ValidLogin) {
  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "valid_login";
  testParams.userInfo.displayName = "Display Name";
  testParams.oauth = "valid_oauth";

  TestChatApiInitializationCallback(TokenizationOptions());

  TTV_ErrorCode ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  ec = LogOut(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);
}

TEST_F(ChatApiTest, InvalidLogIn) {
  TestParams testParams;
  testParams.userInfo.userId = 0;
  testParams.oauth = "";

  TestChatApiInitializationCallback(TokenizationOptions());

  TTV_ErrorCode ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_INVALID_LOGIN, ec);
}

TEST_F(ChatApiTest, InvalidLogOut) {
  TestParams testParams;
  testParams.userInfo.userId = 0;

  TestChatApiInitializationCallback(TokenizationOptions());

  TTV_ErrorCode ec = LogOut(testParams);
  EXPECT_EQ(TTV_EC_INVALID_LOGIN, ec);
}

TEST_F(ChatApiTest, InvalidChannelName) {
  TestParams testParams;
  testParams.userInfo.userId = 9002;
  testParams.userInfo.userName = "valid_login";
  testParams.userInfo.displayName = "Display Name";
  testParams.channelId = 0;
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  EXPECT_EQ(TTV_EC_SUCCESS, LogIn(testParams));

  TTV_ErrorCode invalidChannel = ConnectToChannel(testParams);
  EXPECT_EQ(TTV_EC_INVALID_CHANNEL_ID, invalidChannel);
}

TEST_F(ChatApiTest, ChatChannelConnectDisconnectHappyPath) {
  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "valid_login";
  testParams.userInfo.displayName = "Display Name";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  /*
   * Mock channel properties
   */

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddRequestRegex(std::regex("FetchChannelCheermotes"))
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/pikachu_blocklist_1.json")
    .Done();

  /*
   * Log in
   */
  TTV_ErrorCode ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  /*
   * Attempt to connect to channel
   */

  ec = ConnectToChannel(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  TestChatChannelStateCallback(testParams, ChatChannelState::Connecting, 3000);
  TestIrcHandshake(testParams);
  TestChatChannelStateCallback(testParams, ChatChannelState::Connected, 3000);

  /*
   * On Connection, send simple message
   */

  std::string simpleMsg = "Pokemon.... gotta catch them all";
  SendChatMessage(testParams, simpleMsg);
  TestClientSentPrivMessage(testParams, simpleMsg, 3000);

  /*
   * Attempt disconnect from channel
   */

  ec = DisconnectFromChannel(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);
  TestChatChannelStateCallback(testParams, ChatChannelState::Disconnected, 3000);

  /*
   * Attempt to shut down chat api
   */

  ec = ShutDown();
  EXPECT_EQ(TTV_EC_SUCCESS, ec);
  TestChatApiShutdownCallback();
}

TEST_F(ChatApiTest, TestMultipleUsersInOneChannel) {
  TestChatApiInitializationCallback(TokenizationOptions());

  /*
   * Mock channel properties
   */
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddRequestRegex(std::regex("FetchChannelCheermotes"))
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/pikachu_blocklist_1.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/pikachu_blocklist_1.json")
    .Done();

  TestParams squirtleTestParams;
  squirtleTestParams.oauth = "squirtle_token";
  squirtleTestParams.userInfo.userId = 9001;
  squirtleTestParams.userInfo.userName = "squirtle";
  squirtleTestParams.userInfo.displayName = "Squirtle";
  squirtleTestParams.channelId = 1001;
  squirtleTestParams.channelName = "pokemon";

  TTV_ErrorCode ec = LogIn(squirtleTestParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  ec = ConnectToChannel(squirtleTestParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  TestChatChannelStateCallback(squirtleTestParams, ChatChannelState::Connecting, 3000);
  TestIrcHandshake(squirtleTestParams);
  TestChatChannelStateCallback(squirtleTestParams, ChatChannelState::Connected, 3000);

  TestParams charmanderTestParams;
  charmanderTestParams.oauth = "charmander_token";
  charmanderTestParams.userInfo.userId = 9002;
  charmanderTestParams.userInfo.userName = "charmander";
  charmanderTestParams.userInfo.displayName = "Charmander";
  charmanderTestParams.channelId = 1001;
  charmanderTestParams.channelName = "pokemon";

  ec = LogIn(charmanderTestParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  ec = ConnectToChannel(charmanderTestParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  TestChatChannelStateCallback(charmanderTestParams, ChatChannelState::Connecting, 3000);
  TestIrcHandshake(charmanderTestParams);
  TestChatChannelStateCallback(charmanderTestParams, ChatChannelState::Connected, 3000);

  ec = DisconnectFromChannel(squirtleTestParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);
  TestChatChannelStateCallback(squirtleTestParams, ChatChannelState::Disconnected, 3000);

  ec = DisconnectFromChannel(charmanderTestParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);
  TestChatChannelStateCallback(charmanderTestParams, ChatChannelState::Disconnected, 3000);

  ec = ShutDown();
  EXPECT_EQ(TTV_EC_SUCCESS, ec);
  TestChatApiShutdownCallback();
}

TEST_F(ChatApiTest, Tokenization) {
  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "pikachu";
  testParams.userInfo.displayName = "Pikachu";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  /*
   * Mock channel properties
   */
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddRequestRegex(std::regex("FetchChannelCheermotes"))
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .Done();

  TestChatApiInitializationCallback(TokenizationOptions::All());

  /*
   * Log in
   */
  TTV_ErrorCode ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  /*
   * Attempt to connect to channel
   */

  ConnectToChannel(testParams);
  TestIrcHandshake(testParams);
  TestChatChannelStateCallback(testParams, ChatChannelState::Connected, 3000);

  std::string simpleMsg = "Pokemon, gotta catch them all";
  SendChatMessage(testParams, simpleMsg);
  TestClientSentPrivMessage(testParams, simpleMsg, 3000);

  auto expectedTextToken = std::make_unique<ttv::chat::TextToken>("Pokemon, gotta catch them all");

  ttv::chat::MessageInfo expectedTokenizedMessage;
  expectedTokenizedMessage.userName = "pikachu";
  expectedTokenizedMessage.nameColorARGB = 0xFFFF69B4;
  expectedTokenizedMessage.tokens.emplace_back(std::move(expectedTextToken));

  TestMessageCallback(testParams, {expectedTokenizedMessage}, 6000);

  mChatTestManager->FreeAllMessages();

  TTV_ErrorCode chatApiShutDownSuccessful = ShutDown();
  EXPECT_EQ(TTV_EC_SUCCESS, chatApiShutDownSuccessful);
  TestChatApiShutdownCallback();
}

TEST_F(ChatApiTest, UrlHelper) {
  using namespace ttv;

  {
    Uri url("http://api.twitch.tv:234/blah/a/b/c/d?hi=bye&blah=true");
    ASSERT_EQ(url.GetProtocol(), "http");
    ASSERT_EQ(url.GetHostName(), "api.twitch.tv");
    ASSERT_EQ(url.GetPort(), "234");
    ASSERT_EQ(url.GetPath(), "/blah/a/b/c/d");
    ASSERT_EQ(url.GetParams().size(), 2);
    ASSERT_TRUE(url.GetParams().find("hi") != url.GetParams().end());
    ASSERT_TRUE(url.GetParams().find("hi")->second == "bye");
    ASSERT_TRUE(url.GetParams().find("blah") != url.GetParams().end());
    ASSERT_TRUE(url.GetParams().find("blah")->second == "true");
  }
  {
    Uri url("api.twitch.tv?hi=bye&blah=true");
    ASSERT_EQ(url.GetProtocol(), "");
    ASSERT_EQ(url.GetHostName(), "api.twitch.tv");
    ASSERT_EQ(url.GetPort(), "");
    ASSERT_EQ(url.GetPath(), "");
    ASSERT_EQ(url.GetParams().size(), 2);
    ASSERT_TRUE(url.GetParams().find("hi") != url.GetParams().end());
    ASSERT_TRUE(url.GetParams().find("hi")->second == "bye");
    ASSERT_TRUE(url.GetParams().find("blah") != url.GetParams().end());
    ASSERT_TRUE(url.GetParams().find("blah")->second == "true");
  }
  {
    Uri url("api.twitch.tv/?hi=bye&blah=true");
    ASSERT_EQ(url.GetProtocol(), "");
    ASSERT_EQ(url.GetHostName(), "api.twitch.tv");
    ASSERT_EQ(url.GetPort(), "");
    ASSERT_EQ(url.GetPath(), "/");
    ASSERT_EQ(url.GetParams().size(), 2);
    ASSERT_TRUE(url.GetParams().find("hi") != url.GetParams().end());
    ASSERT_TRUE(url.GetParams().find("hi")->second == "bye");
    ASSERT_TRUE(url.GetParams().find("blah") != url.GetParams().end());
    ASSERT_TRUE(url.GetParams().find("blah")->second == "true");
  }
  {
    Uri url("api.twitch.tv/");
    ASSERT_EQ(url.GetProtocol(), "");
    ASSERT_EQ(url.GetHostName(), "api.twitch.tv");
    ASSERT_EQ(url.GetPort(), "");
    ASSERT_EQ(url.GetPath(), "/");
    ASSERT_EQ(url.GetParams().size(), 0);
  }
  {
    Uri url("api.twitch.tv");
    ASSERT_EQ(url.GetProtocol(), "");
    ASSERT_EQ(url.GetHostName(), "api.twitch.tv");
    ASSERT_EQ(url.GetPort(), "");
    ASSERT_EQ(url.GetPath(), "");
    ASSERT_EQ(url.GetParams().size(), 0);
  }
}

TEST_F(ChatApiTest, FetchUserList) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddRequestRegex(std::regex("FetchChannelCheermotes"))
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .Done();
  mHttpRequest->AddResponse("https://tmi.twitch.tv/group/user/pokemon/chatters")
    .SetResponseBodyFromFile("chat/dragonballzuserlist.json");

  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "pikachu";
  testParams.userInfo.displayName = "Pikachu";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions::All());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  ConnectToChannel(testParams);
  TestIrcHandshake(testParams);
  TestChatChannelStateCallback(testParams, ChatChannelState::Connected, 3000);

  bool callbackReceived = false;
  ec = mChatApi->FetchUserListForChannel(
    testParams.userInfo.userId, testParams.channelId, [&callbackReceived](TTV_ErrorCode ec, UserList&& userList) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      ASSERT_EQ(userList.totalUserCount, 2);

      ASSERT_EQ(userList.moderators.size(), 1);
      ASSERT_EQ(userList.moderators[0], "goku");

      ASSERT_TRUE(userList.staff.empty());
      ASSERT_TRUE(userList.admins.empty());
      ASSERT_TRUE(userList.globalModerators.empty());

      ASSERT_EQ(userList.viewers.size(), 2);
      ASSERT_EQ(userList.viewers[0], "goku");
      ASSERT_EQ(userList.viewers[1], "vegeta");

      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });
}

// TEST_F(ChatApiTest, FetchGlobalBitsConfiguration) {
//  mHttpRequest->AddResponse("https://api.twitch.tv/v5/bits/actions")
//    .SetResponseBodyFromFile("chat/chatgetbitsconfigtask_global.json")
//    .Done();
//
//  TestChatApiInitializationCallback(TokenizationOptions::All());
//
//  bool callbackReceived = false;
//  auto ec = mChatApi->FetchGlobalBitsConfiguration(
//    [&callbackReceived](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
//      ASSERT_TRUE(TTV_SUCCEEDED(ec));
//
//      const std::vector<BitsConfiguration::Cheermote>& cheermotes = config->GetCheermotes();
//
//      ASSERT_EQ(config->GetChannelId(), 0);
//      ASSERT_EQ(cheermotes.size(), 28);
//      ASSERT_EQ(cheermotes[0].prefix, "Cheer");
//      ASSERT_EQ(cheermotes[1].prefix, "DoodleCheer");
//      ASSERT_EQ(cheermotes[2].tiers.size(), 5);
//      ASSERT_EQ(cheermotes[2].tiers[2].bits, 1000);
//      ASSERT_EQ(cheermotes[2].tiers[2].images.size(), 20);
//
//      callbackReceived = true;
//    });
//
//  ASSERT_TRUE(TTV_SUCCEEDED(ec));
//
//  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });
//
//  ASSERT_TRUE(callbackReceived);
//}

TEST_F(ChatApiTest, FetchChannelBitsConfiguration) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddRequestRegex(std::regex("FetchChannelCheermotes"))
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  TestChatApiInitializationCallback(TokenizationOptions::All());

  bool callbackReceived = false;
  auto ec = mChatApi->FetchChannelBitsConfiguration(
    0, 1001, [&callbackReceived](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      const std::vector<BitsConfiguration::Cheermote>& cheermotes = config->GetCheermotes();

      ASSERT_EQ(config->GetChannelId(), 1001);
      ASSERT_EQ(cheermotes.size(), 29);
      ASSERT_EQ(cheermotes[0].prefix, "Cheer");
      ASSERT_EQ(cheermotes[0].type, BitsConfiguration::Cheermote::Type::FirstParty);
      ASSERT_EQ(cheermotes[1].prefix, "DoodleCheer");
      ASSERT_EQ(cheermotes[1].type, BitsConfiguration::Cheermote::Type::ThirdParty);
      ASSERT_EQ(cheermotes[2].prefix, "lirikCheer");
      ASSERT_EQ(cheermotes[2].type, BitsConfiguration::Cheermote::Type::Custom);
      ASSERT_EQ(cheermotes[2].tiers.size(), 5);
      ASSERT_EQ(cheermotes[2].tiers[2].bits, 1000);
      ASSERT_EQ(cheermotes[2].tiers[2].images.size(), 20);

      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, GetEmoticonUrl) {
  TestChatApiInitializationCallback(TokenizationOptions::All());

  std::string url;
  auto ec = mChatApi->GetEmoticonUrl("100", 2.0, url);

  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ASSERT_EQ(url, "https://static-cdn.jtvnw.net/emoticons/v1/100/2.0");
}

TEST_F(ChatApiTest, GetBitsImageUrl) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddRequestRegex(std::regex("FetchChannelCheermotes"))
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  TestChatApiInitializationCallback(TokenizationOptions::All());

  bool callbackReceived = false;
  auto ec = mChatApi->FetchChannelBitsConfiguration(
    0, 1001, [&callbackReceived](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      std::string url;
      ttv::Color color = 0xFF000000;

      ec =
        config->GetBitsImageUrl("cheer", 4999, BitsConfiguration::CheermoteImage::Theme::Dark, 1.0, false, url, color);
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(url, "https://d3aqoihi2n8ty8.cloudfront.net/actions/cheer/dark/static/1000/1.png");
      ASSERT_EQ(color, 4280136357);

      ec = config->GetBitsImageUrl(
        "LIRIKcHeEr", 34, BitsConfiguration::CheermoteImage::Theme::Dark, 1.0, false, url, color);
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(url,
        "https://d3aqoihi2n8ty8.cloudfront.net/partner-actions/23161357/20acbcf5-59e0-4f3f-ab93-0ea54628b542/1/light/static/1/ce458256c237bfddf963bd28cdc0f4da836ebddb.png");
      ASSERT_EQ(color, 4288124823);

      ec =
        config->GetBitsImageUrl("KAPPA", 5001, BitsConfiguration::CheermoteImage::Theme::Light, 3.0, true, url, color);
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(url, "https://d3aqoihi2n8ty8.cloudfront.net/actions/kappa/light/animated/5000/3.gif");
      ASSERT_EQ(color, 4278229502);

      ec = config->GetBitsImageUrl("chee", 34, BitsConfiguration::CheermoteImage::Theme::Dark, 1.0, false, url, color);
      ASSERT_EQ(ec, TTV_EC_NOT_AVAILABLE);
      ASSERT_TRUE(url.empty());

      ec =
        config->GetBitsImageUrl("cheere", 34, BitsConfiguration::CheermoteImage::Theme::Dark, 1.0, false, url, color);
      ASSERT_EQ(ec, TTV_EC_NOT_AVAILABLE);
      ASSERT_TRUE(url.empty());

      ec = config->GetBitsImageUrl("cheer", 34, BitsConfiguration::CheermoteImage::Theme::Dark, 5.0, false, url, color);
      ASSERT_EQ(ec, TTV_EC_NOT_AVAILABLE);
      ASSERT_TRUE(url.empty());

      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, GetHighestDpiBitsImageUrl) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddRequestRegex(std::regex("FetchChannelCheermotes"))
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  TestChatApiInitializationCallback(TokenizationOptions::All());

  bool callbackReceived = false;
  auto ec = mChatApi->FetchChannelBitsConfiguration(
    0, 1001, [&callbackReceived](TTV_ErrorCode ec, const std::shared_ptr<BitsConfiguration>& config) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));

      std::string url;
      ttv::Color color = 0xFF000000;

      ec = config->GetHighestDpiBitsImageUrl(
        "cheer", 4999, BitsConfiguration::CheermoteImage::Theme::Dark, 1.0, false, url, color);
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(url, "https://d3aqoihi2n8ty8.cloudfront.net/actions/cheer/dark/static/1000/1.png");
      ASSERT_EQ(color, 4280136357);

      ec = config->GetHighestDpiBitsImageUrl(
        "LIRIKcHeEr", 34, BitsConfiguration::CheermoteImage::Theme::Dark, 1.5, false, url, color);
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(url,
        "https://d3aqoihi2n8ty8.cloudfront.net/partner-actions/23161357/20acbcf5-59e0-4f3f-ab93-0ea54628b542/1/light/static/1.5/3aa334b8d33c6f54e84f8fbf48d593a1b3b890fb.png");
      ASSERT_EQ(color, 4288124823);

      ec = config->GetHighestDpiBitsImageUrl(
        "KAPPA", 5000, BitsConfiguration::CheermoteImage::Theme::Light, 5.0, true, url, color);
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(url, "https://d3aqoihi2n8ty8.cloudfront.net/actions/kappa/light/animated/5000/4.gif");
      ASSERT_EQ(color, 4278229502);

      ec = config->GetHighestDpiBitsImageUrl(
        "cheer", 34, BitsConfiguration::CheermoteImage::Theme::Dark, 0.0, false, url, color);
      ASSERT_EQ(ec, TTV_EC_NOT_AVAILABLE);
      ASSERT_TRUE(url.empty());

      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, GetChannelVodCommentSettings) {
  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/videos/channels/12826/settings")
    .SetType(ttv::HTTP_GET_REQUEST)
    .SetResponseBodyFromFile("chat/chatchannelvodcommentsettingstask.json")
    .Done();

  TestChatApiInitializationCallback(TokenizationOptions::All());

  // Log a viewer in
  TestParams testParams;
  testParams.userInfo.userId = 12826;
  testParams.userInfo.userName = "valid_login";
  testParams.userInfo.displayName = "Display Name";
  testParams.oauth = "valid_oauth";

  TTV_ErrorCode ec = LogIn(testParams);

  ASSERT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->FetchChannelVodCommentSettings(
    12826, 12826, [&callbackReceived](TTV_ErrorCode ec, ChannelVodCommentSettings&& settings) {
      callbackReceived = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      ASSERT_EQ(settings.channelId, 12826);
      ASSERT_EQ(settings.publishingMode, CommentPublishingMode::Disabled);
      ASSERT_EQ(settings.createdAt, 1505859963);
      ASSERT_EQ(settings.updatedAt, 1505859963);
      ASSERT_EQ(settings.followersOnlyDurationSeconds, 86400);
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, SetChannelVodCommentSettings) {
  mHttpRequest->AddResponse("https://api.twitch.tv/kraken/videos/channels/12826/settings")
    .SetType(ttv::HTTP_PUT_REQUEST)
    .SetResponseBodyFromFile("chat/chatchannelvodcommentsettingstask.json")
    .Done();

  TestChatApiInitializationCallback(TokenizationOptions::All());

  // Log a viewer in
  TestParams testParams;
  testParams.userInfo.userId = 12826;
  testParams.userInfo.userName = "valid_login";
  testParams.userInfo.displayName = "Display Name";
  testParams.oauth = "valid_oauth";

  TTV_ErrorCode ec = LogIn(testParams);

  ASSERT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->SetChannelVodPublishingMode(
    12826, 12826, CommentPublishingMode::Disabled, [&callbackReceived](TTV_ErrorCode ec) {
      callbackReceived = true;
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  callbackReceived = false;
  ec = mChatApi->SetChannelVodFollowersOnlyDuration(12826, 12826, 86400, [&callbackReceived](TTV_ErrorCode ec) {
    callbackReceived = true;
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, ModUser) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatmoduser.json")
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->ModUser(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, ModUserError&& /*error*/) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, UnmodUser) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatunmoduser.json")
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->UnmodUser(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, UnmodUserError&& /*error*/) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, ModUser_Error) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatmoduser_error.json");

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->ModUser(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, ModUserError&& error) {
    ASSERT_FALSE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error.code, GraphQLErrorCode::TARGET_ALREADY_MOD);
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, UnmodUser_Error) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatunmoduser_error.json");

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->UnmodUser(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, UnmodUserError&& error) {
    ASSERT_FALSE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error.code, GraphQLErrorCode::TARGET_NOT_MOD);
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, BanUser) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatbanuser.json")
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->BanUser(1001, 1001, "otherUser", 500, [&callbackReceived](TTV_ErrorCode ec, BanUserError&& error) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error.code, GraphQLErrorCode::SUCCESS);
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, UnbanUser) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatunbanuser.json")
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->UnbanUser(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, UnbanUserError&& error) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error.code, GraphQLErrorCode::SUCCESS);
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, BanUser_Error) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatbanuser_error.json")
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->BanUser(1001, 1001, "otherUser", 500, [&callbackReceived](TTV_ErrorCode ec, BanUserError&& error) {
    ASSERT_FALSE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error.code, GraphQLErrorCode::TARGET_IS_BROADCASTER);
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, UnbanUser_Error) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatunbanuser_error.json")
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->UnbanUser(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, UnbanUserError&& error) {
    ASSERT_FALSE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error.code, GraphQLErrorCode::TARGET_NOT_BANNED);
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, ListModerators) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatlistmoderators.json")
    .Done();

  TestChatApiInitializationCallback(TokenizationOptions());

  bool callbackReceived = false;
  auto ec = mChatApi->FetchChannelModerators(1001, "",
    [&callbackReceived](TTV_ErrorCode ec, const std::vector<std::string>& modNames, const std::string& nextCursor) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      callbackReceived = true;

      ASSERT_EQ(modNames.size(), 100);
      ASSERT_EQ(modNames[0], "Twitch");
      ASSERT_EQ(nextCursor, "1473551396533821000");
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, ChatRestrictions) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "pikachu";
  testParams.userInfo.displayName = "Pikachu";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  ConnectToChannel(testParams);
  TestIrcHandshake(testParams);
  TestChatChannelStateCallback(testParams, ChatChannelState::Connected, 3000);

  ChatChannelInfo chatChannelInfo;
  ChatChannelRestrictions chatChannelRestrictions;

  mTransport->EnqueueIncomingData(
    "@broadcaster-lang=en;emote-only=1;followers-only=0;mercury=0;r9k=1;room-id=1001;slow=600;subs-only=1 :tmi.twitch.tv ROOMSTATE #pokemon\r\n");
  {
    auto checkFunc = [this, &testParams, &chatChannelRestrictions]() {
      chatChannelRestrictions =
        mChatTestManager->GetChatChannelRestrictionsFor(testParams.userInfo.userId, testParams.channelId);
      return (chatChannelRestrictions.slowModeDuration == 600);
    };
    WaitUntilResult(2000, checkFunc);
    ASSERT_TRUE(checkFunc());
    ASSERT_TRUE(chatChannelRestrictions.emoteOnly);
    ASSERT_TRUE(chatChannelRestrictions.followersOnly);
    ASSERT_EQ(chatChannelRestrictions.followersDuration, 0);
    ASSERT_TRUE(chatChannelRestrictions.r9k);
    ASSERT_TRUE(chatChannelRestrictions.slowMode);
    ASSERT_EQ(chatChannelRestrictions.slowModeDuration, 600);
    ASSERT_TRUE(chatChannelRestrictions.subscribersOnly);
  }

  mTransport->EnqueueIncomingData(
    "@broadcaster-lang=;emote-only=0;followers-only=-1;mercury=0;r9k=0;room-id=1001;slow=0;subs-only=0 :tmi.twitch.tv ROOMSTATE #pokemon\r\n");
  {
    auto checkFunc = [this, &testParams, &chatChannelRestrictions]() {
      chatChannelRestrictions =
        mChatTestManager->GetChatChannelRestrictionsFor(testParams.userInfo.userId, testParams.channelId);
      return chatChannelRestrictions.slowModeDuration == 0;
    };
    WaitUntilResult(2000, checkFunc);
    ASSERT_TRUE(checkFunc());
    ASSERT_FALSE(chatChannelRestrictions.emoteOnly);
    ASSERT_FALSE(chatChannelRestrictions.followersOnly);
    ASSERT_EQ(chatChannelRestrictions.followersDuration, 0);
    ASSERT_FALSE(chatChannelRestrictions.r9k);
    ASSERT_FALSE(chatChannelRestrictions.slowMode);
    ASSERT_EQ(chatChannelRestrictions.slowModeDuration, 0);
    ASSERT_FALSE(chatChannelRestrictions.subscribersOnly);
  }

  mTransport->EnqueueIncomingData("@followers-only=600 :tmi.twitch.tv ROOMSTATE #pokemon\r\n");
  {
    auto checkFunc = [this, &testParams, &chatChannelRestrictions]() {
      chatChannelRestrictions =
        mChatTestManager->GetChatChannelRestrictionsFor(testParams.userInfo.userId, testParams.channelId);
      return (chatChannelRestrictions.followersOnly && (chatChannelRestrictions.followersDuration == 600));
    };
    WaitUntilResult(2000, checkFunc);
    ASSERT_TRUE(checkFunc());
  }
}

TEST_F(ChatApiTest, UserEmoticonSets) {
  std::shared_ptr<MockResponse> response = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                                             .SetType(ttv::HTTP_POST_REQUEST)
                                             .SetStatusCode(200)
                                             .SetResponseBodyFromFile("chat/chatuseremoticons.json")
                                             .Done();

  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "pikachu";
  testParams.userInfo.displayName = "Pikachu";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions::All());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  auto checkFunc = [this, &testParams]() {
    return mChatTestManager->GetChatUserEmoticonSets(testParams.userInfo.userId).size() > 0;
  };

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunc, GetDefaultUpdateFunc()));

  const std::vector<EmoticonSet>& emoticonSets = mChatTestManager->GetChatUserEmoticonSets(testParams.userInfo.userId);
  ASSERT_EQ(emoticonSets.size(), 1);

  EmoticonSet set = emoticonSets[0];
  ASSERT_EQ(set.emoticonSetId, "0");
  ASSERT_EQ(set.emoticons.size(), 208);

  response->SetResponseBodyFromFile("chat/chatuseremoticons2.json");

  ttv::json::Value pubsubMessage;
  pubsubMessage["user_id"] = "9001";
  pubsubMessage["channel_id"] = "1001";
  mPubSubTestUtility.PushPubSubMessage("user-subscribe-events-v1.9001", pubsubMessage);

  auto checkFunc2 = [this, &testParams]() {
    return mChatTestManager->GetChatUserEmoticonSets(testParams.userInfo.userId).size() > 1;
  };

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, checkFunc2, GetDefaultUpdateFunc()));
  const std::vector<EmoticonSet>& emoticonSets2 = mChatTestManager->GetChatUserEmoticonSets(testParams.userInfo.userId);
  ASSERT_EQ(emoticonSets2.size(), 2);

  set = emoticonSets2[0];
  ASSERT_EQ(set.emoticonSetId, "19194");
  ASSERT_EQ(set.emoticons.size(), 30);

  set = emoticonSets2[1];
  ASSERT_EQ(set.emoticonSetId, "0");
  ASSERT_EQ(set.emoticons.size(), 207);
}

TEST_F(ChatApiTest, GrantVIP) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatgrantvip.json")
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec =
    mChatApi->GrantVIP(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, GrantVIPErrorCode&& /*error*/) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, RevokeVIP) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatrevokevip.json")
    .Done();

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec =
    mChatApi->RevokeVIP(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, RevokeVIPErrorCode&& /*error*/) {
      ASSERT_TRUE(TTV_SUCCEEDED(ec));
      callbackReceived = true;
    });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, GrantVIP_Error) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatgrantvip_error.json");

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->GrantVIP(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, GrantVIPErrorCode error) {
    ASSERT_FALSE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error, GrantVIPErrorCode::GRANTEE_ALREADY_VIP);
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, RevokeVIP_Error) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatrevokevip_error.json");

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->RevokeVIP(1001, 1001, "otherUser", [&callbackReceived](TTV_ErrorCode ec, RevokeVIPErrorCode error) {
    ASSERT_FALSE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(error, RevokeVIPErrorCode::REVOKEE_NOT_VIP);
    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, ListVIPs) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBodyFromFile("chat/chatlistvips.json");

  TestParams testParams;
  testParams.userInfo.userId = 1001;
  testParams.userInfo.userName = "pokemon";
  testParams.userInfo.displayName = "Pokemon";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->FetchChannelVIPs(1001, [&callbackReceived](TTV_ErrorCode ec, std::vector<std::string>&& vipNames) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    callbackReceived = true;

    ASSERT_EQ(vipNames.size(), 2);
    ASSERT_EQ(vipNames[0], "user1");
    ASSERT_EQ(vipNames[1], "user2");
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, FetchGlobalBadges) {
  auto payload = R"(
  {
    "data": {
      "badges": [
        {
          "clickURL": "click_url_1",
          "description": "description_1",
          "imageUrlNormal": "image_url_1_1",
          "imageUrlDouble": "image_url_2_1",
          "imageUrlQuadruple": "image_url_4_1",
          "onClickAction": "SUBSCRIBE",
          "setID": "set_id_1",
          "title": "title_1",
          "version": "version_1"
        },
        {
          "clickURL": "click_url_2",
          "description": "description_2",
          "imageUrlNormal": "image_url_1_2",
          "imageUrlDouble": "image_url_2_2",
          "imageUrlQuadruple": "image_url_4_2",
          "onClickAction": "GET_TURBO",
          "setID": "set_id_2",
          "title": "title_2",
          "version": "version_2"
        },
        {
          "clickURL": "click_url_3",
          "description": "description_3",
          "imageUrlNormal": "image_url_1_3",
          "imageUrlDouble": "image_url_2_3",
          "imageUrlQuadruple": "image_url_4_3",
          "onClickAction": "GET_BITS",
          "setID": "set_id_3",
          "title": "title_3",
          "version": "version_3"
        },
        {
          "clickURL": "click_url_4",
          "description": "description_4",
          "imageUrlNormal": "image_url_1_4",
          "imageUrlDouble": "image_url_2_4",
          "imageUrlQuadruple": "image_url_4_4",
          "onClickAction": "VISIT_URL",
          "setID": "set_id_4",
          "title": "title_4",
          "version": "version_4"
        }
      ]
    }
  }
  )";

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBody(payload)
    .AddRequestRegex(std::regex("FetchGlobalBadges"));

  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "pikachu";
  testParams.userInfo.displayName = "Pikachu";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions::All());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->FetchGlobalBadges([&callbackReceived](TTV_ErrorCode ec, BadgeSet&& badgeSet) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    EXPECT_EQ(4, badgeSet.badges.size());
    std::map<int, BadgeVersion::Action> expectedActions = {
      // clang-format off
      {1, BadgeVersion::Action::Subscribe},
      {2, BadgeVersion::Action::GetTurbo},
      {3, BadgeVersion::Action::GetBits},
      {4, BadgeVersion::Action::VisitUrl}
      // clang-format on
    };

    for (int i = 1; i <= 4; i++) {
      auto badge = badgeSet.badges[ConcatValue("set_id_", i)];
      EXPECT_EQ(badge.name, ConcatValue("set_id_", i));
      auto versionBadge = badge.versions[ConcatValue("version_", i)];

      EXPECT_EQ(versionBadge.clickUrl, ConcatValue("click_url_", i));
      EXPECT_EQ(versionBadge.name, ConcatValue("version_", i));
      EXPECT_EQ(versionBadge.title, ConcatValue("title_", i));
      EXPECT_EQ(versionBadge.description, ConcatValue("description_", i));
      EXPECT_EQ(versionBadge.clickAction, expectedActions[i]);

      EXPECT_EQ(versionBadge.images[0].url, ConcatValue("image_url_1_", i));
      EXPECT_EQ(versionBadge.images[0].scale, 1.0f);

      EXPECT_EQ(versionBadge.images[1].url, ConcatValue("image_url_2_", i));
      EXPECT_EQ(versionBadge.images[1].scale, 2.0f);

      EXPECT_EQ(versionBadge.images[2].url, ConcatValue("image_url_4_", i));
      EXPECT_EQ(versionBadge.images[2].scale, 4.0f);

      BadgeImage result;

      versionBadge.FindImage(0.0f, result);
      EXPECT_EQ(versionBadge.images[0].url, result.url);

      versionBadge.FindImage(1.0f, result);
      EXPECT_EQ(versionBadge.images[0].url, result.url);

      versionBadge.FindImage(2.0f, result);
      EXPECT_EQ(versionBadge.images[1].url, result.url);

      versionBadge.FindImage(4.0f, result);
      EXPECT_EQ(versionBadge.images[2].url, result.url);

      versionBadge.FindImage(10.0f, result);
      EXPECT_EQ(versionBadge.images[2].url, result.url);
    }

    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, FetchChannelBadges) {
  auto payload = R"(
  {
    "data": {
        "user": {
        "broadcastBadges": [
          {
            "clickURL": "click_url_1",
            "description": "description_1",
            "imageUrlNormal": "image_url_1_1",
            "imageUrlDouble": "image_url_2_1",
            "imageUrlQuadruple": "image_url_4_1",
            "onClickAction": "SUBSCRIBE",
            "setID": "set_id_1",
            "title": "title_1",
            "version": "version_1"
          },
          {
            "clickURL": "click_url_2",
            "description": "description_2",
            "imageUrlNormal": "image_url_1_2",
            "imageUrlDouble": "image_url_2_2",
            "imageUrlQuadruple": "image_url_4_2",
            "onClickAction": "GET_TURBO",
            "setID": "set_id_2",
            "title": "title_2",
            "version": "version_2"
          },
          {
            "clickURL": "click_url_3",
            "description": "description_3",
            "imageUrlNormal": "image_url_1_3",
            "imageUrlDouble": "image_url_2_3",
            "imageUrlQuadruple": "image_url_4_3",
            "onClickAction": "GET_BITS",
            "setID": "set_id_3",
            "title": "title_3",
            "version": "version_3"
          },
          {
            "clickURL": "click_url_4",
            "description": "description_4",
            "imageUrlNormal": "image_url_1_4",
            "imageUrlDouble": "image_url_2_4",
            "imageUrlQuadruple": "image_url_4_4",
            "onClickAction": "VISIT_URL",
            "setID": "set_id_4",
            "title": "title_4",
            "version": "version_4"
          }
        ]
      }
    }
  }
  )";

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetResponseBody(payload)
    .AddRequestRegex(std::regex("FetchChannelBadges"))
    .AddJsonValue({"variables", "channelId"}, "98765");

  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "pikachu";
  testParams.userInfo.displayName = "Pikachu";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions::All());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool callbackReceived = false;
  ec = mChatApi->FetchChannelBadges(98765, [&callbackReceived](TTV_ErrorCode ec, BadgeSet&& badgeSet) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));

    EXPECT_EQ(4, badgeSet.badges.size());
    std::map<int, BadgeVersion::Action> expectedActions = {
      // clang-format off
      {1, BadgeVersion::Action::Subscribe},
      {2, BadgeVersion::Action::GetTurbo},
      {3, BadgeVersion::Action::GetBits},
      {4, BadgeVersion::Action::VisitUrl}
      // clang-format on
    };

    for (int i = 1; i <= 4; i++) {
      auto badge = badgeSet.badges[ConcatValue("set_id_", i)];
      EXPECT_EQ(badge.name, ConcatValue("set_id_", i));
      auto versionBadge = badge.versions[ConcatValue("version_", i)];

      EXPECT_EQ(versionBadge.clickUrl, ConcatValue("click_url_", i));
      EXPECT_EQ(versionBadge.name, ConcatValue("version_", i));
      EXPECT_EQ(versionBadge.title, ConcatValue("title_", i));
      EXPECT_EQ(versionBadge.description, ConcatValue("description_", i));
      EXPECT_EQ(versionBadge.clickAction, expectedActions[i]);

      EXPECT_EQ(versionBadge.images[0].url, ConcatValue("image_url_1_", i));
      EXPECT_EQ(versionBadge.images[0].scale, 1.0f);

      EXPECT_EQ(versionBadge.images[1].url, ConcatValue("image_url_2_", i));
      EXPECT_EQ(versionBadge.images[1].scale, 2.0f);

      EXPECT_EQ(versionBadge.images[2].url, ConcatValue("image_url_4_", i));
      EXPECT_EQ(versionBadge.images[2].scale, 4.0f);

      BadgeImage result;

      versionBadge.FindImage(0.0f, result);
      EXPECT_EQ(versionBadge.images[0].url, result.url);

      versionBadge.FindImage(1.0f, result);
      EXPECT_EQ(versionBadge.images[0].url, result.url);

      versionBadge.FindImage(2.0f, result);
      EXPECT_EQ(versionBadge.images[1].url, result.url);

      versionBadge.FindImage(4.0f, result);
      EXPECT_EQ(versionBadge.images[2].url, result.url);

      versionBadge.FindImage(10.0f, result);
      EXPECT_EQ(versionBadge.images[2].url, result.url);
    }

    callbackReceived = true;
  });

  ASSERT_TRUE(TTV_SUCCEEDED(ec));

  WaitUntilResult(1000, [&callbackReceived]() { return callbackReceived; });

  ASSERT_TRUE(callbackReceived);
}

TEST_F(ChatApiTest, WhisperMessage) {
  const auto payload = R"(
    {
      "type": "whisper_received",
      "data_object":
      {
        "body": "iiiiuuuu",
        "from_id": 22222,
        "sent_ts": 1443745783,
        "tags":
        {
            "color": "#5F9EA0",
            "display_name": "test_user_name_1",
            "emotes": [],
            "login": "test_user_name_1",
            "turbo": true,
            "user_type": ""
        },
        "id": 1,
        "thread_id": "thread_id_1",
        "message_id": "message_id_1"
      }
    }
  )";

  TestParams testParams;
  testParams.userInfo.userId = 9001;
  testParams.userInfo.userName = "pikachu";
  testParams.userInfo.displayName = "Pikachu";
  testParams.channelId = 1001;
  testParams.channelName = "pokemon";
  testParams.oauth = "auth_token";

  TestChatApiInitializationCallback(TokenizationOptions::All());

  auto ec = LogIn(testParams);
  EXPECT_EQ(TTV_EC_SUCCESS, ec);

  bool ready = false;
  auto listener = std::make_shared<ChatUserThreadsProxyTester>();
  listener->mChatThreadRealtimeMessageReceived = [&ready](ttv::UserId userId, const std::string& threadId,
                                                   const WhisperMessage& messageList) {
    EXPECT_EQ(messageList.messageId, 1);
    EXPECT_EQ(messageList.threadId, "thread_id_1");
    EXPECT_EQ(messageList.messageUuid, "message_id_1");
    ready = true;
  };

  mChatApi->SetUserThreadsListener(9001, listener);

  mPubSubTestUtility.PushPubSubMessage("whispers.9001", payload);

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000, [&ready] { return ready; }, GetDefaultUpdateFunc()));
}
