/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "chattestmanager.h"
#include "fixtures/chatapitest.h"
#include "testchatobjectfactory.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichatobjectfactory.h"
#include "twitchsdk/chat/internal/chatuserblocklist.h"
#include "twitchsdk/core/task/taskrunner.h"
#include "twitchsdk/core/thread.h"
#include "twitchsdk/core/user/user.h"
#include "twitchsdk/core/user/userrepository.h"

#include <thread>

#include "gtest/gtest.h"

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::chat::test;

TEST_F(ChatBaseTest, UserBlockList) {
  std::shared_ptr<TaskRunner> taskRunner = std::make_shared<TaskRunner>();

  std::shared_ptr<UserRepository> userRepository = std::make_shared<UserRepository>();
  userRepository->SetTaskRunner(taskRunner);
  ttv::test::InitializeComponent(userRepository);

  UserInfo userInfo;
  userInfo.userId = 9001;
  userInfo.userName = "pikachu";
  userInfo.displayName = "Pikachu";

  auto user = userRepository->RegisterUser(userInfo.userId);
  user->SetUserInfo(userInfo);

  std::shared_ptr<ChatUserBlockList> blockList = std::make_shared<ChatUserBlockList>(user);
  blockList->SetTaskRunner(taskRunner);

  std::shared_ptr<MockResponse> mock = mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
                                         .SetType(ttv::HTTP_POST_REQUEST)
                                         .SetResponseBodyFromFile("chat/pikachu_blocklist_1.json")
                                         .Done();
  std::unordered_set<UserId> blockedUserIds;

  std::function<void()> updateFunc = [userRepository, blockList, taskRunner]() {
    taskRunner->PollTasks();
    blockList->Update();
    userRepository->Update();
  };

  // Get the initial list
  std::function<bool()> checkFunc = [&blockedUserIds, blockList]() {
    blockList->GetBlockedUsers(blockedUserIds);
    return blockedUserIds.size() == 1;
  };

  ttv::test::InitializeComponent(blockList);

  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, checkFunc, updateFunc));
  mock->AssertRequestsMade();
  ASSERT_EQ(blockedUserIds.size(), 1);

  // Fetch Blocked users
  int numCallbacksCalled = 0;
  checkFunc = [&numCallbacksCalled]() { return numCallbacksCalled == 3; };

  auto fetchBlockedUsersCallback = [&numCallbacksCalled](TTV_ErrorCode ec, const std::vector<UserInfo>& users) {
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ASSERT_EQ(users.size(), 1);
    ASSERT_EQ(users.at(0).userId, 45168716);
    ASSERT_EQ(users.at(0).userName, "dude");

    numCallbacksCalled++;
  };

  blockList->FetchBlockedUsers(fetchBlockedUsersCallback);
  blockList->FetchBlockedUsers(fetchBlockedUsersCallback);
  blockList->FetchBlockedUsers(fetchBlockedUsersCallback);
  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, checkFunc, updateFunc));

  // Ignore someone
  checkFunc = [&blockedUserIds, blockList]() {
    blockList->GetBlockedUsers(blockedUserIds);
    return blockedUserIds.size() == 2;
  };

  mock = mHttpRequest->AddResponse("https://api.twitch.tv/kraken/users/9001/blocks/333")
           .SetType(HTTP_PUT_REQUEST)
           .AddRequestParam("reason", "spite")
           .AddRequestParam("whisper", "false")
           .SetResponseBodyFromFile("chat/pikachu_block.json")
           .Done();
  ASSERT_EQ(blockList->BlockUser(333, "spite", false), TTV_EC_SUCCESS);
  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, checkFunc, updateFunc));
  mock->AssertRequestsMade();
  ASSERT_EQ(blockedUserIds.size(), 2);

  // Unignore the user
  checkFunc = [&blockedUserIds, blockList]() {
    blockList->GetBlockedUsers(blockedUserIds);
    return blockedUserIds.size() == 1;
  };

  mock = mHttpRequest->AddResponse("https://api.twitch.tv/kraken/users/9001/blocks/333")
           .SetType(HTTP_DELETE_REQUEST)
           .SetResponseBodyFromFile("chat/pikachu_unblock.json")
           .Done();
  ASSERT_EQ(blockList->UnblockUser(333), TTV_EC_SUCCESS);
  EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(1000, checkFunc, updateFunc));
  mock->AssertRequestsMade();
  blockList->GetBlockedUsers(blockedUserIds);
  ASSERT_EQ(blockedUserIds.size(), 1);

  // Cleanup
  ttv::test::ShutdownComponent(blockList);
  ttv::test::ShutdownComponent(userRepository);
  ttv::test::ShutdownTaskRunner(taskRunner);
}
