/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichatchannel.h"
#include "twitchsdk/chat/ichatchannelproperties.h"
#include "twitchsdk/chat/ichatchannelpropertylistener.h"
#include "twitchsdk/chat/internal/chatcommentmanager.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"

#include <iostream>

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
constexpr const char* kTopicPrefix = "stream-chat-room-v1.12345";
constexpr char kChatRestrictionGql[] = R"({
  "data": {
    "user": {
      "chatSettings": {
        "autoModLevel": 0,
        "blockLinks": false,
        "chatDelayMs": 6000,
        "followersOnlyDurationMinutes": null,
        "isBroadcasterLanguageModeEnabled": false,
        "isFastSubsModeEnabled": false,
        "isOptedOutOfGlobalBannedWordsList": false,
        "isSubscribersOnlyModeEnabled": false,
        "requireVerifiedAccount": false,
        "rules": [
          "Test"
        ],
        "slowModeDurationSeconds": null
      }
    }
  },
  "extensions": {
    "durationMilliseconds": 48,
    "operationName": "testing"
  }
})";

std::string BuildChatRestrictionsJson(const ChatChannelRestrictions& restrictions) {
  std::stringstream ss;

  ss << R"({"type":"updated_room")"
     << R"(,"data":{)"
     << R"("room":{)"
     << R"("modes":{)"
     << R"("followers_only_duration_minutes":)" << restrictions.followersDuration << R"(,"emote_only_mode_enabled":)"
     << (restrictions.emoteOnly ? "true" : "false") << R"(,"subscribers_only_mode_enabled":)"
     << (restrictions.subscribersOnly ? "true" : "false") << R"(,"verified_only_mode_enabled":)"
     << (restrictions.verifiedOnly ? "true" : "false") << R"(,"r9k_mode_enabled":)"
     << (restrictions.r9k ? "true" : "false") << R"(,"slow_mode_duration_seconds":)" << restrictions.slowModeDuration
     << R"(,"slow_mode_set_at":")" << UnixTimestampToRFC3339String(restrictions.slowModeSetAt) << R"(")"
     << R"(}}}})";

  return ss.str();
}

class ChatChannelPropertiesTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions::All());

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "test";
    mUserInfo.displayName = "Test";

    ASSERT_TRUE(TTV_SUCCEEDED(LogIn(mCoreApi, "auth_token", mUserInfo)));

    mListener = std::make_shared<ChatChannelPropertyListenerProxy>();
    ASSERT_TRUE(TTV_SUCCEEDED(mChatApi->CreateChatChannelProperties(0, 12345, mListener, mChatChannelProperties)));

    ASSERT_TRUE(WaitUntilResultWithPollTask(
      1000, [this]() { return mPubSubTestUtility.IsSubscribedToTopic(kTopicPrefix); }, GetDefaultUpdateFunc()));
  }

  bool IsCallbackReceived() {
    bool receivedCallback = false;
    mListener->chatChannelRestrictionsReceivedCallback =
      [&receivedCallback](ChatChannelRestrictions&& /*chatChannelRestrictions*/) { receivedCallback = true; };

    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc());

    return receivedCallback;
  }

  ChatChannelRestrictions CaptureRestrictionsCallback() {
    bool receivedCallback = false;
    ChatChannelRestrictions ret;
    mListener->chatChannelRestrictionsReceivedCallback = [&receivedCallback, &ret](
                                                           ChatChannelRestrictions&& chatChannelRestrictions) {
      ret = chatChannelRestrictions;
      receivedCallback = true;
    };

    EXPECT_TRUE(
      WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

    return ret;
  }

  ChatChannelRestrictions HelperPubAndReceiveRestrictions(const std::string& payload) {
    mPubSubTestUtility.PushPubSubMessage(kTopicPrefix, payload);

    return CaptureRestrictionsCallback();
  }

  ChatChannelRestrictions HelperPubAndReceiveRestrictions(const ChatChannelRestrictions& restrictions) {
    return HelperPubAndReceiveRestrictions(BuildChatRestrictionsJson(restrictions));
  }

  virtual void TearDownComponents() override { SdkBaseTest::TearDownComponents(); }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<ChatAPI> mChatApi;
  std::shared_ptr<IChatChannelProperties> mChatChannelProperties;
  std::shared_ptr<ChatChannelPropertyListenerProxy> mListener;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};
}  // namespace

TEST_F(ChatChannelPropertiesTest, ChatRestrictions) {
  ChatChannelRestrictions expected;

  expected.followersDuration = 10;
  expected.slowModeDuration = 20;
  expected.slowModeSetAt = 30;

  auto result = HelperPubAndReceiveRestrictions(expected);

  EXPECT_EQ(result.followersDuration, 10);
  EXPECT_EQ(result.slowModeDuration, 20);
  EXPECT_EQ(result.slowModeSetAt, 30);

  EXPECT_FALSE(result.emoteOnly);
  expected.emoteOnly = true;
  result = HelperPubAndReceiveRestrictions(expected);
  EXPECT_TRUE(result.emoteOnly);

  EXPECT_FALSE(result.verifiedOnly);
  expected.verifiedOnly = true;
  result = HelperPubAndReceiveRestrictions(expected);
  EXPECT_TRUE(result.verifiedOnly);

  EXPECT_FALSE(result.subscribersOnly);
  expected.subscribersOnly = true;
  result = HelperPubAndReceiveRestrictions(expected);
  EXPECT_TRUE(result.subscribersOnly);

  EXPECT_FALSE(result.r9k);
  expected.r9k = true;
  result = HelperPubAndReceiveRestrictions(expected);
  EXPECT_TRUE(result.r9k);
}

TEST_F(ChatChannelPropertiesTest, ChatRestrictionsGqlInit) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBody(kChatRestrictionGql)
    .Done();
  EXPECT_EQ(IsCallbackReceived(), true);

  mPubSubTestUtility.PushPubSubMessage(kTopicPrefix, BuildChatRestrictionsJson({}));
  EXPECT_EQ(IsCallbackReceived(), true);
}

TEST_F(ChatChannelPropertiesTest, ChatRestrictionsPubSubInit) {
  mPubSubTestUtility.PushPubSubMessage(kTopicPrefix, BuildChatRestrictionsJson({}));
  EXPECT_EQ(IsCallbackReceived(), true);

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBody(kChatRestrictionGql)
    .Done();
  EXPECT_EQ(IsCallbackReceived(), false);
}

TEST_F(ChatChannelPropertiesTest, ChatRestrictionsDependantValues) {
  ChatChannelRestrictions expected;

  expected.followersDuration = 0;
  auto result = HelperPubAndReceiveRestrictions(expected);
  EXPECT_TRUE(result.followersOnly);

  expected.followersDuration = 10;
  result = HelperPubAndReceiveRestrictions(expected);
  EXPECT_TRUE(result.followersOnly);

  expected.slowModeDuration = 0;
  result = HelperPubAndReceiveRestrictions(expected);
  EXPECT_FALSE(result.slowMode);

  expected.slowModeDuration = 10;
  result = HelperPubAndReceiveRestrictions(expected);
  EXPECT_TRUE(result.slowMode);
}

TEST_F(ChatChannelPropertiesTest, GqlSlowDurationLessThanZero) {
  constexpr char payload[] = R"({
  "data": {
    "user": {
      "chatSettings": {
        "autoModLevel": 0,
        "blockLinks": false,
        "chatDelayMs": 6000,
        "followersOnlyDurationMinutes": null,
        "isBroadcasterLanguageModeEnabled": false,
        "isFastSubsModeEnabled": false,
        "isOptedOutOfGlobalBannedWordsList": false,
        "isSubscribersOnlyModeEnabled": false,
        "requireVerifiedAccount": false,
        "rules": [
          "Test"
        ],
        "slowModeDurationSeconds": -1
      }
    }
  },
  "extensions": {
    "durationMilliseconds": 48,
    "operationName": "testing"
  }
})";
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBody(payload)
    .Done();

  auto restrictions = CaptureRestrictionsCallback();
  EXPECT_FALSE(restrictions.slowMode);
  EXPECT_EQ(restrictions.slowModeDuration, 0);
}

TEST_F(ChatChannelPropertiesTest, PubsubSlowDurationLessThanZero) {
  ChatChannelRestrictions expected;
  expected.slowModeDuration = static_cast<uint32_t>(-1);

  auto restrictions = HelperPubAndReceiveRestrictions(expected);
  EXPECT_FALSE(restrictions.slowMode);
  EXPECT_EQ(restrictions.slowModeDuration, 0);
}
