/****************************************************************************
 * Twitch SDK
 *
 * This software is supplied under the terms of a license agreement with
 * Twitch Interactive, Inc. and may not be copied or used except in accordance
 * with the terms of that agreement
 *
 * Copyright (c) 2012-2016 Twitch Interactive, Inc.
 ***************************************************************************/

#include "fixtures/chatapitest.h"
#include "fixtures/sdkbasetest.h"
#include "testutilities.h"
#include "twitchsdk/chat/chatapi.h"
#include "twitchsdk/chat/ichatchannel.h"
#include "twitchsdk/chat/internal/chatcommentmanager.h"
#include "twitchsdk/core/coreapi.h"
#include "twitchsdk/core/stringutilities.h"

#include <iostream>

using namespace ttv;
using namespace ttv::chat;
using namespace ttv::test;
using namespace ttv::chat::test;

namespace {
class ChatChannelTest : public SdkBaseTest {
 public:
  virtual void SetUpComponents() override {
    SdkBaseTest::SetUpComponents();

    mCoreApi = std::make_shared<CoreAPI>();
    mChatApi = std::make_shared<ChatAPI>();

    mChatApi->SetCoreApi(mCoreApi);
    mChatApi->SetTokenizationOptions(TokenizationOptions::All());

    mTransport = std::make_shared<TestTransport>();
    mTestFactory = std::make_shared<TestChatObjectFactory>();

    mTestFactory->SetTransport(std::static_pointer_cast<IChatTransport>(mTransport));
    mChatApi->SetChatObjectFactory(std::static_pointer_cast<IChatObjectFactory>(mTestFactory));

    InitializeModule(mCoreApi);
    InitializeModule(mChatApi);

    AddModule(mCoreApi);
    AddModule(mChatApi);

    mUserInfo.userId = 12345;
    mUserInfo.userName = "test";
    mUserInfo.displayName = "Test";

    TTV_ErrorCode ec = LogIn(mCoreApi, "auth_token", mUserInfo);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
  }

  virtual void TearDownComponents() override { SdkBaseTest::TearDownComponents(); }

 protected:
  UserInfo mUserInfo;
  std::shared_ptr<ChatAPI> mChatApi;
  std::shared_ptr<TestChatObjectFactory> mTestFactory;
  std::shared_ptr<TestTransport> mTransport;

 private:
  std::shared_ptr<CoreAPI> mCoreApi;
};

class ChatChannelListener : public ttv::chat::IChatChannelListener {
 public:
  using ChatChannelModNoticeMessageDeletedCallback = std::function<void(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message)>;
  using ChatChannelLocalUserChangedCallback =
    std::function<void(UserId userId, ChannelId channelId, const ChatUserInfo& userInfo)>;

  virtual void ChatChannelStateChanged(
    UserId userId, ChannelId channelId, ChatChannelState state, TTV_ErrorCode ec) override {}

  virtual void ChatChannelInfoChanged(UserId userId, ChannelId channelId, const ChatChannelInfo& info) override {}

  virtual void ChatChannelRestrictionsChanged(
    UserId userId, ChannelId channelId, const ChatChannelRestrictions& restrictions) override {}

  virtual void ChatChannelLocalUserChanged(UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) override {
    if (chatChannelLocalUserChangedCallback != nullptr) {
      chatChannelLocalUserChangedCallback(userId, channelId, userInfo);
    }
  }

  virtual void ChatChannelMessagesReceived(
    UserId userId, ChannelId channelId, const std::vector<LiveChatMessage>& messageList) override {}

  virtual void ChatChannelSubscriptionNoticeReceived(
    UserId userId, ChannelId channelId, const SubscriptionNotice& notice) override {}

  virtual void ChatChannelFirstTimeChatterNoticeReceived(
    UserId userId, ChannelId channelId, const FirstTimeChatterNotice& notice) override {}

  virtual void ChatChannelRaidNoticeReceived(UserId userId, ChannelId channelId, const RaidNotice& notice) override {}

  virtual void ChatChannelUnraidNoticeReceived(
    UserId userId, ChannelId channelId, const UnraidNotice& notice) override {}

  virtual void ChatChannelGenericNoticeReceived(
    UserId userId, ChannelId channelId, const GenericMessageNotice& notice) override {}

  virtual void ChatChannelMessagesCleared(UserId userId, ChannelId channelId) override {}

  virtual void ChatChannelUserMessagesCleared(UserId userId, ChannelId channelId, UserId clearUserId) override {}

  virtual void ChatChannelMessageDeleted(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& senderLoginName, std::string&& messageContent) override {}

  virtual void ChatChannelHostTargetChanged(
    UserId userId, ChannelId channelId, const std::string& targetChannel, uint32_t /*numViewers*/) override {}

  virtual void ChatChannelNoticeReceived(UserId userId, ChannelId channelId, const std::string& id,
    const std::map<std::string, std::string>& params) override {}

  virtual void AutoModCaughtSentMessage(UserId userId, ChannelId channelId) override {}

  virtual void AutoModDeniedSentMessage(UserId userId, ChannelId channelId) override {}

  virtual void AutoModApprovedSentMessage(UserId userId, ChannelId channelId) override {}

  virtual void AutoModCaughtMessageForMods(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) override {}

  virtual void AutoModMessageApprovedByMod(UserId userId, ChannelId channelId, std::string&& messageId,
    UserId moderatorId, std::string&& moderatorName) override {}

  virtual void AutoModMessageDeniedByMod(UserId userId, ChannelId channelId, std::string&& messageId,
    UserId moderatorId, std::string&& moderatorName) override {}

  virtual void AutoModDeniedSentCheer(UserId userId, ChannelId channelId) override {}

  virtual void AutoModTimedOutSentCheer(UserId userId, ChannelId channelId) override {}

  virtual void AutoModCaughtCheerForMods(UserId userId, ChannelId channelId, std::string&& messageId,
    std::string&& message, UserId senderId, std::string&& senderName, std::string&& reason) override {}

  virtual void ChatChannelModNoticeUserTimedOut(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, uint32_t timeoutDurationSeconds, std::string&& reason) override {}

  virtual void ChatChannelModNoticeUserBanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo, std::string&& reason) override {}

  virtual void ChatChannelModNoticeUserUntimedOut(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) override {}

  virtual void ChatChannelModNoticeUserUnbanned(
    UserId userId, ChannelId channelId, ModerationActionInfo&& modActionInfo) override {}

  virtual void ChatChannelModNoticeClearChat(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeEmoteOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeEmoteOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeFollowersOnly(UserId userId, ChannelId channelId, UserId modId,
    std::string&& modName, uint32_t minimumFollowingDurationMinutes) override {}

  virtual void ChatChannelModNoticeFollowersOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeR9K(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeR9KOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeSlow(UserId userId, ChannelId channelId, UserId modId, std::string&& modName,
    uint32_t slowModeDurationSeconds) override {}

  virtual void ChatChannelModNoticeSlowOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeSubsOnly(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeSubsOnlyOff(
    UserId userId, ChannelId channelId, UserId modId, std::string&& modName) override {}

  virtual void ChatChannelModNoticeMessageDeleted(UserId userId, ChannelId channelId,
    ModerationActionInfo&& modActionInfo, std::string&& messageId, std::string&& message) override {
    if (chatChannelModNoticeMessageDeletedCallback != nullptr) {
      chatChannelModNoticeMessageDeletedCallback(
        userId, channelId, std::move(modActionInfo), std::move(messageId), std::move(message));
    }
  }

  ChatChannelModNoticeMessageDeletedCallback chatChannelModNoticeMessageDeletedCallback;
  ChatChannelLocalUserChangedCallback chatChannelLocalUserChangedCallback;
};
}  // namespace

TEST_F(ChatChannelTest, CreateChatChannel) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .Done();

  std::shared_ptr<IChatChannelListener> listener = std::make_shared<ChatChannelListener>();
  std::shared_ptr<IChatChannel> chatChannel;

  for (int i = 0; i < 10; i++) {
    TTV_ErrorCode ec = mChatApi->CreateChatChannel(12345, 1001, listener, chatChannel);
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ec = chatChannel->Connect();

    std::function<bool()> transportOpenCheckFunc = [this]() { return mTransport->IsOpen(); };

    EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(5000, transportOpenCheckFunc, GetDefaultUpdateFunc()));

    ASSERT_TRUE(mTransport->IsOpen());
    ASSERT_TRUE(TTV_SUCCEEDED(ec));
    ec = chatChannel->Disconnect();
    chatChannel.reset();

    std::function<bool()> transportClosedCheckFunc = [this]() { return !mTransport->IsOpen(); };

    EXPECT_TRUE(ttv::test::WaitUntilResultWithPollTask(5000, transportClosedCheckFunc, GetDefaultUpdateFunc()));
  }
}

TEST_F(ChatChannelTest, ModNoticeMessageDeleted) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .Done();

  bool receivedCallback = false;
  std::shared_ptr<ChatChannelListener> listener = std::make_shared<ChatChannelListener>();
  listener->chatChannelModNoticeMessageDeletedCallback = [&receivedCallback](UserId userId, ChannelId channelId,
                                                           ModerationActionInfo&& modActionInfo,
                                                           std::string&& messageId, std::string&& message) {
    EXPECT_EQ(userId, 12345);
    EXPECT_EQ(channelId, 1001);
    EXPECT_EQ(messageId, "messageId");
    EXPECT_EQ(message, "bad message");
    EXPECT_EQ(modActionInfo.moderatorId, 1001);
    EXPECT_EQ(modActionInfo.targetId, 1002);
    EXPECT_EQ(modActionInfo.moderatorName, "moderator");
    EXPECT_EQ(modActionInfo.targetName, "sender");
    receivedCallback = true;
  };
  std::shared_ptr<IChatChannel> chatChannel;
  TTV_ErrorCode ec = mChatApi->CreateChatChannel(12345, 1001, listener, chatChannel);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ec = chatChannel->Connect();

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("chat_moderator_actions.12345.1001"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("chat_moderator_actions.12345.1001",
    "{\"type\":\"moderation_action\",\"data\":{\"type\":\"chat_login_moderation\",\"moderation_action\":\"delete\",\"args\":[\"sender\",\"bad message\",\"messageId\"],\"created_by\":\"moderator\",\"created_by_user_id\":\"1001\",\"msg_id\":\"\",\"target_user_id\":\"1002\"}}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}

TEST_F(ChatChannelTest, UserInfoUpdated) {
  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatfetchchannelcheermotes.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetType(ttv::HTTP_POST_REQUEST)
    .AddJsonValue({"variables", "userID"}, "1001")
    .SetStatusCode(200)
    .SetResponseBodyFromFile("chat/chatapi_channel_info.json")
    .Done();

  mHttpRequest->AddResponse("https://gql.twitch.tv/gql")
    .SetResponseBodyFromFile("chat/pokemonchatproperties.json")
    .SetType(ttv::HTTP_POST_REQUEST)
    .Done();

  bool receivedCallback = false;
  std::shared_ptr<ChatChannelListener> listener = std::make_shared<ChatChannelListener>();
  listener->chatChannelLocalUserChangedCallback = [&receivedCallback](
                                                    UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) {
    receivedCallback = true;
    EXPECT_EQ(userId, 12345);
    EXPECT_EQ(channelId, 1001);
    EXPECT_TRUE(userInfo.userMode.moderator);
    EXPECT_FALSE(userInfo.userMode.vip);
  };

  std::shared_ptr<IChatChannel> chatChannel;
  TTV_ErrorCode ec = mChatApi->CreateChatChannel(12345, 1001, listener, chatChannel);
  ASSERT_TRUE(TTV_SUCCEEDED(ec));
  ec = chatChannel->Connect();

  ASSERT_TRUE(WaitUntilResultWithPollTask(1000,
    [this]() { return mPubSubTestUtility.IsSubscribedToTopic("chat_moderator_actions.12345.1001"); },
    GetDefaultUpdateFunc()));

  mPubSubTestUtility.PushPubSubMessage("chat_moderator_actions.12345.1001",
    "{\"type\":\"moderator_added\",\"data\":{\"channel_id\":\"1001\",\"target_user_id\":\"12345\"}}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

  listener->chatChannelLocalUserChangedCallback = [&receivedCallback](
                                                    UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) {
    receivedCallback = true;
    EXPECT_EQ(userId, 12345);
    EXPECT_EQ(channelId, 1001);
    EXPECT_FALSE(userInfo.userMode.moderator);
    EXPECT_TRUE(userInfo.userMode.vip);
  };

  receivedCallback = false;
  mPubSubTestUtility.PushPubSubMessage("chat_moderator_actions.12345.1001",
    "{\"type\":\"vip_added\",\"data\":{\"channel_id\":\"1001\",\"target_user_id\":\"12345\"}}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));

  listener->chatChannelLocalUserChangedCallback = [&receivedCallback](
                                                    UserId userId, ChannelId channelId, const ChatUserInfo& userInfo) {
    receivedCallback = true;
    EXPECT_EQ(userId, 12345);
    EXPECT_EQ(channelId, 1001);
    EXPECT_FALSE(userInfo.userMode.moderator);
    EXPECT_FALSE(userInfo.userMode.vip);
  };

  receivedCallback = false;
  mPubSubTestUtility.PushPubSubMessage("chat_moderator_actions.12345.1001",
    "{\"type\":\"vip_removed\",\"data\":{\"channel_id\":\"1001\",\"target_user_id\":\"12345\"}}");

  ASSERT_TRUE(
    WaitUntilResultWithPollTask(1000, [&receivedCallback]() { return receivedCallback; }, GetDefaultUpdateFunc()));
}
